/*
 *  Module:  implicit curvature lines
 *
 *  Program: impcurv (implicit curvature lines generator for project Riemann)
 *
 *  Author:  Jim Lambers
 *
 *  Purpose: mesh (small.step mesh.frequency initial.point d1 d2 human.equation)
 *           compute orthogonal mesh for the implicit case on human.equation
 *           calls curvature direction code
 *           d1 = max distance in cd1 direction
 *           d2 = max distance in cd2 direction
 */

#define	MAIN
#include "implicit.h"

main (argc, argv)
int	argc;
char	*argv[];
{
    int 	i;
    SCALAR	scD1Steps, 	/* Length of direction 1 & 2 lines, in steps */
                scD2Steps;
    int		D1Lines, 	/* Number of direction 1 & 2 lines */
                D2Lines;
    SCALAR      scSpacing, scD1, scD2;
    SCALAR	scStep, cdgu1(), cdgu2();
    VECTOR	vctPoint, vctZero;
    char	*sEqn, *sTitle;


    peqnSurface = (PEQN) MyMalloc(sizeof (EQN));

    UserInterface(argc, argv, &vctPoint, &scStep, &scSpacing, &scD1, &scD2,
      peqnSurface, &pfOutfile, &sTitle, &sEqn);

    scD1Steps = scD1 / scStep;
    scD2Steps = scD2 / scStep;
    D1Lines = (int) (scD2 / scSpacing) + 1;
    D2Lines = (int) (scD1 / scSpacing) + 1;

    /* Allocate memory for the points, allowing one extra VECTOR in each */
    /*   dimension to help avoid unspotted off-by-one errors causing */
    /*   segmentation faults */
    avctCD1Lines = (PVECTOR) MyMalloc( (D1Lines + 1) * ((int) scD1Steps + 2) * 
      sizeof(VECTOR) );
    avctCD2Lines = (PVECTOR) MyMalloc( (D2Lines + 1) * ((int) scD2Steps + 2) *
      sizeof(VECTOR) );

    for (i = 0; i < NUMBER_OF_VARS; i++) {
	apeqnPartials[i] = PartialEqn(CopyEqn(peqnSurface), i);
	vctZero.vars[i] = 0.0;
    }

    MakeHess();

    cd(scD2, NUMBER_OF_VARS, scStep, cdgu2, &vctPoint, &vctZero, 
      &vctGlobalOldCD, avctCD2Lines);
    cd(scD1, NUMBER_OF_VARS, scStep, cdgu1, &vctPoint, &vctZero, 
      &vctGlobalOldCD, avctCD1Lines);

    mesh(scD1, scD2Steps, scSpacing, cdgu1, scStep, avctCD1Lines,
	avctCD2Lines, &vctGlobalOldCD);
    mesh(scD2, scD1Steps, scSpacing, cdgu2, scStep, avctCD2Lines,
	avctCD1Lines, &vctGlobalOldCD);
    
    outputcds(&vctPoint, scStep, scSpacing, scD1, scD2, IMP_CURV, sTitle, 
      peqnSurface->et, &sEqn);
}



/*
 *   Function:  outputcds
 *
 *   Purpose:   This function outputs the header and all of the points to the
 *		output file.
 *
 *   Author:    Bret Johnson
 */

void	outputcds(pvctInitial, scStep, scSpacing, scD1, scD2, type, sTitle, et,
  sEqn)
PVECTOR pvctInitial;
SCALAR	scStep;
SCALAR	scSpacing, scD1, scD2;
int	type;
char	*sTitle;
EQNTYPE	et;
char	*sEqn[];
{
    int		i;
    int         D1Points,            /* Number of points computed in */
                D2Points;            /*   directions one and two */
    int		D1Lines, 	     /* Number of lines computed in */
		D2Lines;	     /*   directions one and two */


    /* Output standard header information */
    OutputHeader(pfOutfile, type, sTitle, et, sEqn, FALSE);

    /* Output initial point in header */
    fprintf(pfOutfile, "Initial point: (");
    for (i = 0; i < NUMBER_OF_VARS; ++i) {
        fprintf(pfOutfile, "%.10lf", pvctInitial->vars[i]);

        if (i != NUMBER_OF_VARS - 1)
            fprintf(pfOutfile, ", ");
        else fprintf(pfOutfile, ")\n");
    }

    /* Output other miscellaneous header information */
    fprintf(pfOutfile, "Step size: %f\n", scStep);
    fprintf(pfOutfile, "Mesh spacing: %f\n", scSpacing);
    fprintf(pfOutfile, "Direction one distance: %f\n", scD1);
    fprintf(pfOutfile, "Direction two distance: %f\n\n", scD2);

    D1Points = (int) (scD1 / scStep) + 1;
    D2Points = (int) (scD2 / scStep) + 1;
    D1Lines = (int) (scD2 / scSpacing) + 1;
    D2Lines = (int) (scD1 / scSpacing) + 1;

    /* Output the points along the direction 1 and 2 curvature lines */
    outputpts(D1Points, D1Lines, avctCD1Lines, type, 1);
    outputpts(D2Points, D2Lines, avctCD2Lines, type, 2);
}


 
/*
 *   Function: outputpts
 *
 *   Purpose:  This function outputs all of the computed points along the 
 *	       curvature lines in one direction.
 *
 *   Author:   Jim Lambers
 *
 */

void	outputpts(NumOfPoints, NumOfLines, avctLines, type, direction)
int	NumOfPoints, NumOfLines;
VECTOR	avctLines[];
{
    int		i, j;


    for (i = 0; i < NumOfLines; i++) {
	OutputLineStart(pfOutfile, type, direction);
        for (j = 0; j < NumOfPoints; j++)
	    OutputPoint(pfOutfile, &(avctLines[NumOfPoints*i + j]), 3);
    }
}



/*
 *   Function:	cdgu1
 *
 *   Author:	Jim Lambers
 *
 */

SCALAR	cdgu1(index, pvctL)
int	index;
PVECTOR	pvctL;
{
    int		i;
    VECTOR	vctOldCD, vctCD1, vctDirection;

    
    switch (index) {

	case 0:
 	    VctCopy(&vctOldCD, &vctGlobalOldCD);
	    for (i = 0; i < NUMBER_OF_VARS; i++)
		vctCD1.vars[i] = 0.0;
	    CurvatureDirections(1, pvctL, &vctGlobalOldCD, &vctDirection);
	    PossReverseCurvature(&vctDirection, &vctOldCD, &vctCD1);
	    VctCopy(&vctGlobalOldCD, &vctCD1);
	    return(vctCD1.vars[0]);
	    break;

	case 1: 
	    return(vctGlobalOldCD.vars[1]);
    }

    return(vctGlobalOldCD.vars[2]);
}



/*
 *   Function:	cdgu2
 *
 *   Author:	Jim Lambers
 *
 */

SCALAR	cdgu2(index, pvctL)
int	index;
PVECTOR	pvctL;
{
    int		i;
    VECTOR	vctOldCD, vctCD2, vctDirection;

    
    switch (index) {

	case 0:
	    VctCopy(&vctOldCD, &vctGlobalOldCD);
	    for (i = 0; i < NUMBER_OF_VARS; i++)
		vctCD2.vars[i] = 0.0;
	    CurvatureDirections(2, pvctL, &vctGlobalOldCD, &vctDirection);
	    PossReverseCurvature(&vctDirection, &vctOldCD, &vctCD2);
	    VctCopy(&vctGlobalOldCD, &vctCD2);
	    return(vctCD2.vars[0]);
	    break;

	case 1: 
	    return(vctGlobalOldCD.vars[1]);
    }

    return(vctGlobalOldCD.vars[2]);
}



/*
 *   Function:	HessianatPoint
 *
 *   Author:	Jim Lambers
 *
 */

void	HessianatPoint(pvctPoint)
PVECTOR	pvctPoint;
{
    int	i = 0, j;


    for (; i < 3; i++)
	for (j = 0; j < 3; j++)
	    ascResultMatrix[i][j] = EvaluateEqn(apeqnHess[i][j], pvctPoint);
}



/*
 *   Function:	CurvatureDirections
 *
 *   Author:	Jim Lambers
 *
 */

void	CurvatureDirections(flag, pvctPoint, pvctOldCD, pvctDirection)
int	flag;
PVECTOR	pvctPoint, pvctOldCD, pvctDirection;
{
    SCALAR	scW11 = 0.0, scW12 = 0.0, scW22 = 0.0, ascA[2][2], scFLength,
    		scEv1, scX1, scX2, scK, scBigh, scX1New, scX2New;
    VECTOR	vctV1, vctV2, vctF, vctFNorm, vctRes1, vctRes2, vctProjection;
    int		i, j;


    EqnGradient(apeqnPartials, pvctPoint, &vctF);
    Normalize(&vctF, &vctFNorm);
    ComputeOrthogonalBasis(&vctFNorm, &vctV1, &vctV2);
    
    HessianatPoint(pvctPoint);
    
    for (i = 0; i < 3; i++)
	for (j = 0; j < 3; j++)
	    scW11 += (ascResultMatrix[i][j] * vctV1.vars[i] * vctV1.vars[j]);
    /* printf("%lf\n", scW11);  */
    scFLength = VctDotVct(&vctF, &vctF);
    
    scW11 = - (scW11 / sqrt(scFLength)); 

    for (i = 0; i < 3; i++)
	for (j = 0; j < 3; j++)
	    scW12 += (ascResultMatrix[i][j] * vctV1.vars[i] * vctV2.vars[j]);
    
    /* printf("%lf\n", scW12);  */
    scW12 = - (scW12 / sqrt(scFLength));
    
    for (i = 0; i < 3; i++)
	for (j = 0; j < 3; j++)
	    scW22 += (ascResultMatrix[i][j] * vctV2.vars[i] * vctV2.vars[j]);
    
    /* printf("%lf\n", scW22); */
    scW22 = - (scW22 / sqrt(scFLength));

    scK = (scW11 * scW22) - (scW12 * scW12);
    
    scEv1 = scBigh = (scW11 + scW22) / 2;
    if (flag == 1)
	scEv1 += (sqrt((scBigh * scBigh) - scK));
    else
	scEv1 -= (sqrt((scBigh * scBigh) - scK));
    
    ascA[0][0] = scW11 - scEv1;
    ascA[0][1] = ascA[1][0] = scW12;
    ascA[1][1] = scW22 - scEv1;

    if (umbilicp(ascA)) {
	projectnormal(pvctOldCD, pvctPoint, &vctProjection);
	Normalize(&vctProjection, pvctOldCD);
	if (!(IsZeroVct(pvctOldCD))) 
	    fprintf(stderr, "Choose a different starting point\n");
    }

    else {
	if ( (fabs(ascA[0][0]) + fabs(ascA[0][1])) > 
          (fabs(ascA[1][0]) + fabs(ascA[1][1])) ) {
	    scX1 = - (ascA[0][1]);
	    scX2 = ascA[0][0];
	}
	else {
	    scX1 = - (ascA[1][1]);
	    scX2 = ascA[1][0];
	}
	scX1New = scX1 / sqrt((scX1 * scX1) + (scX2 * scX2));
	scX2New = scX2 / sqrt((scX1 * scX1) + (scX2 * scX2));
	VctTimesSc(&vctV1, scX1New, &vctRes1);
	VctTimesSc(&vctV2, scX2New, &vctRes2);
	VctPlusVct(&vctRes1, &vctRes2, pvctDirection);
	if (flag != 1) Normalize(pvctDirection, pvctDirection);
    }
}



/*
 *   Function:	MakeHess
 *
 *   Author:	Jim Lambers
 *
 */

void	MakeHess()
{
    int 	i = 0, j;
    PEQN	peqnPartialCopy;


    for (; i < 3; i++) 
	for (j = 0; j < 3; j++) {
	    peqnPartialCopy = CopyEqn(apeqnPartials[i]);
	    apeqnHess[i][j] = PartialEqn(peqnPartialCopy, j);
	}
}



/*
 *   Function:	ComputeOrthogonalBasis
 *
 *   Author:	Jim Lambers
 *
 */

void	ComputeOrthogonalBasis(pvctF, pvctV1, pvctV2)
PVECTOR pvctF, pvctV1, pvctV2;
{
    VECTOR	vctW1, vctNormF, vctCross, vctTemp;


    vctW1.vars[0] = pvctF->vars[1];
    vctW1.vars[1] = -(pvctF->vars[0]);
    vctW1.vars[2] = 0.0;
    if ((fabs(pvctF->vars[2])) > (fabs(pvctF->vars[0]))) {
	vctTemp.vars[0] = 0.0;
	vctTemp.vars[1] = pvctF->vars[2];
	vctTemp.vars[2] = -(pvctF->vars[1]);
	Normalize(&vctTemp, &vctW1);
    } 
    Normalize(&vctW1, pvctV1);
    Normalize(pvctF, &vctNormF);
    VctCrossproduct(&vctW1, &vctNormF, &vctCross);
    Normalize(&vctCross, pvctV2);
}



/*
 *   Function: projectnormal
 *
 *   Author:   Jim Lambers
 */

void	projectnormal(pvctCurrent, pvctPoint, pvctProjection)
PVECTOR	pvctCurrent, pvctPoint,	pvctProjection;
{
    SCALAR	scDotProd;
    VECTOR	vctNormal, vctTemp, vctGradient;

   
    EqnGradient(apeqnPartials, pvctPoint, &vctGradient);
    Normalize(&vctGradient, &vctNormal);
    scDotProd = VctDotVct(pvctCurrent, &vctNormal);
    VctTimesSc(&vctNormal, scDotProd, &vctTemp);
    VctMinusVct(pvctCurrent, &vctTemp, pvctProjection);
}



/*
 *   Function:	PossReverseCurvature
 *
 *   Author:	Jim Lambers
 */

void	PossReverseCurvature(pvctCD1, pvctOldCD, pvctResult)
PVECTOR	pvctCD1, pvctOldCD, pvctResult;
{
    int 	i;
    SCALAR	scTest1 = 0.0, scTest2 = 0.0;
    VECTOR	vctTemp1, vctTemp2, vctTemp3;

  
    if (IsZeroVct(pvctOldCD) && IsZeroVct(pvctCD1))
	fprintf(stderr, "Both *pvctOldCD and *pvctCD1 are nil\n");
    else if (!(IsZeroVct(pvctOldCD))) {
	NegateVct(pvctCD1, &vctTemp1);
	VctMinusVct(&vctTemp1, pvctOldCD, &vctTemp2);
	for (i = 0; i < NUMBER_OF_VARS; i++)
	    scTest1 += fabs(vctTemp2.vars[i]);
	VctMinusVct(pvctCD1, pvctOldCD, &vctTemp3);
	for (i = 0; i < NUMBER_OF_VARS; i++)
	    scTest2 += fabs(vctTemp3.vars[i]);
	if (scTest1 < scTest2) NegateVct(pvctCD1, pvctCD1);
    }

    VctCopy(pvctResult, pvctCD1);
}



/*
 *   Function:	umbilicp
 *
 *   Author:	Jim Lambers
 */

umbilicp(ascA)
SCALAR	ascA[2][2];
{
    SCALAR	scSum = 0.0;
    int 	i, j;


    for (i = 0; i < 2; i++)
	for (j = 0; j < 2; j++)
	    scSum += fabs(ascA[i][j]);

    return(scSum < 0.0001);
}

