/* 
 *  Module:   parametric curvature lines
 *
 *  Programs: parcurv (parametric curvature lines generator for project Riemann)
 */

#define	MAIN
#include "parametric.h"

/*
 *   Program: Paramatric Curvature Lines
 *
 *   Purpose: mesh (small.step mesh.frequency initial.point d1 d2 
 *              human.equation)
 *            compute orthogonal mesh for the implicit case on human.equation
 *            calls curvature direction code
 *            d1 = max distance in cd1 direction
 *            d2 = max distance in cd2 direction
 *  
 *   Author:  Jim Lambers
 *
 */

main(argc, argv) 
int	argc;
char	*argv[]; 
{
    int		i, j;
    SCALAR      scD1Steps,      /* Length of direction 1 & 2 lines, in steps */
                scD2Steps;
    int         D1Lines,        /* Number of direction 1 & 2 lines */
                D2Lines;
    SCALAR	scSpacing, scD1, scD2;
    SCALAR 	scStep, cdpgu1(), cdpgu2(); 
    VECTOR	vctPoint, vctZero;
    char	*asEqn[3], *sTitle;
#ifdef MOEBIUS
    VECTOR	vctTemp;
#endif


    for (i = 0; i < 3; i++)
	apeqnSurface[i] = (PEQN) MyMalloc(sizeof(EQN));

    UserInterface(argc, argv, &vctPoint, &scStep, &scSpacing, &scD1, &scD2,
      apeqnSurface, &pf2dOutfile, &pf3dOutfile, &sTitle, asEqn);

    scD1Steps = scD1 / scStep;
    scD2Steps = scD2 / scStep;
    D1Lines = (int) (scD2 / scSpacing) + 1;
    D2Lines = (int) (scD1 / scSpacing) + 1;

    /* Allocate memory for the points, allowing one extra VECTOR in each */
    /*   dimension to help avoid unspotted off-by-one errors causing */
    /*   segmentation faults */
    avctCD1Lines = (PVECTOR) MyMalloc( (D1Lines + 1) * ((int) scD1Steps + 2) *
      sizeof(VECTOR) );
    avctCD2Lines = (PVECTOR) MyMalloc( (D2Lines + 1) * ((int) scD2Steps + 2) *
      sizeof(VECTOR) );

    for (i = 0; i < NUMBER_OF_VARS; i++)
	vctZero.vars[i] = 0.0;
    
    for (j = 0; j < 2; j++)
	for (i = 0; i < 3; i++)
	    apeqnXSVVF[j][i] = PartialEqn(CopyEqn(apeqnSurface[i]), j);

    for (i = 0; i < 2; i++)
	for (j = 0; j < 2; j++)
	    apeqnGsMatrix[i][j] = vvfinnerproduct(apeqnXSVVF[i], apeqnXSVVF[j]);
    
    for (i = 0; i < 3; i++) {
	apeqnX00[i] = PartialEqn(CopyEqn(apeqnXSVVF[0][i]), 0);
	apeqnX01[i] = PartialEqn(CopyEqn(apeqnXSVVF[0][i]), 1);
	apeqnX11[i] = PartialEqn(CopyEqn(apeqnXSVVF[1][i]), 1);
    }

    cd(scD2, NUMBER_OF_VARS, scStep, cdpgu2, &vctPoint, &vctZero, 
      &vctGlobalOldCD, avctCD2Lines);
    cd(scD1, NUMBER_OF_VARS, scStep, cdpgu1, &vctPoint, &vctZero, 
      &vctGlobalOldCDP, avctCD1Lines);

#ifdef MOEBIUS
    for (i = 0; i < 200; i++) {
	if (fabs(avctCD1Lines[i].vars[0]) <= 0.1) {
	    EvaluateParEqn(apeqnSurface, &(avctCD1Lines[i]), &vctTemp);
	    OutputPoint(stderr, &vctTemp);
	}
    }
#endif

    mesh(scD1, scD2Steps, scSpacing, cdpgu1, scStep, avctCD1Lines, avctCD2Lines,
      &vctGlobalOldCDP);

#ifdef MOEBIUS
    for (i = 0; i < 200; i++) {
	if (fabs(avctCD2Lines[i].vars[0]) <= 0.1) {
	    EvaluateParEqn(apeqnSurface, &(avctCD2Lines[i]), &vctTemp);
	    OutputPoint(stderr, &vctTemp);
	}
    }
#endif

    mesh(scD2, scD1Steps, scSpacing, cdpgu2, scStep, avctCD2Lines, avctCD1Lines,
      &vctGlobalOldCD);
    
    outputcds(&vctPoint, scStep, scSpacing, scD1, scD2, PAR_CURV, sTitle,
        apeqnSurface[0]->et, asEqn);
}



/*
 *   Function: outputcds
 *
 *   Purpose:  This function outputs the headers and all of the points to the
 *	       two output files.
 *
 *   Author:   Bret Johnson
 */

void	outputcds(pvctInitial, scStep, scSpacing, scD1, scD2, type, sTitle, et,
  asEqn)
PVECTOR pvctInitial;
SCALAR	scStep;
SCALAR	scSpacing, scD1, scD2;
int	type;
char	*sTitle;
EQNTYPE	et;
char	*asEqn[];
{
    int         D1Points,            /* Number of points computed in */
                D2Points;            /*   directions one and two */
    int         D1Lines,             /* Number of lines computed in */
                D2Lines;             /*   directions one and two */


    /* Output standard header information to the 2D and 3D output files */
    OutputHeader(pf2dOutfile, type, sTitle, et, asEqn, TRUE);
    OutputHeader(pf3dOutfile, type, sTitle, et, asEqn, FALSE);

    /* Output miscellaneous header information to the 2D and 3D output files */
    OutputMiscHeaderInfo(pf2dOutfile, pvctInitial, scStep, scSpacing, scD1, 
      scD2);
    OutputMiscHeaderInfo(pf3dOutfile, pvctInitial, scStep, scSpacing, scD1, 
      scD2);

    D1Points = (int) (scD1 / scStep) + 1;
    D2Points = (int) (scD2 / scStep) + 1;
    D1Lines = (int) (scD2 / scSpacing) + 1;
    D2Lines = (int) (scD1 / scSpacing) + 1;

    /* Output the points along the direction 1 and 2 curvature lines */
    outputpts(D1Points, D1Lines, avctCD1Lines, type, 1);
    outputpts(D2Points, D2Lines, avctCD2Lines, type, 2);
}



/*
 *   Function: OutputMiscHeaderInfo
 *
 *   Purpose:  This function outputs the passed miscellaneous header information
 *	       to file pfOutfile.
 *
 *   Author:   Bret Johnson
 */

void    OutputMiscHeaderInfo(pfOutfile, pvctInitial, scStep, scSpacing, scD1, 
  scD2)
FILE	* pfOutfile;
PVECTOR	  pvctInitial;
SCALAR	  scStep;
SCALAR	  scSpacing, scD1, scD2;
{
    int		i;


    /* Output initial parameters in header */
    fprintf(pfOutfile, "Initial parameters: (");
    for (i = 0; i < NUMBER_OF_VARS - 1; ++i) {
        fprintf(pfOutfile, "%.10lf", pvctInitial->vars[i]);

        if (i != NUMBER_OF_VARS - 2)
            fprintf(pfOutfile, ", ");
        else fprintf(pfOutfile, ")\n");
    }

    /* Output other miscellaneous header information */
    fprintf(pfOutfile, "Step size: %lf\n", scStep);
    fprintf(pfOutfile, "Mesh spacing: %f\n", scSpacing);
    fprintf(pfOutfile, "Direction one distance: %f\n", scD1);
    fprintf(pfOutfile, "Direction two distance: %f\n\n", scD2);
}


 
/*
 *   Function: outputpts
 *
 *   Purpose:  This function outputs all of the points to the 2D and 3D output
 *	       files.
 *
 *   Author:   Jim Lambers
 */

void	outputpts(NumOfPoints, NumOfLines, avctLines, type, direction)
int	NumOfPoints, NumOfLines;
VECTOR	avctLines[];
{
    int		i, j;
    VECTOR	vctTemp;


    for (i = 0; i < NumOfLines; i++) {

	OutputLineStart(pf2dOutfile, type, direction);
	OutputLineStart(pf3dOutfile, type, direction);

        for (j = 0; j < NumOfPoints; j++) {

#ifdef MOEBIUS
	    if (fabs(avctLines[NumOfPoints*i + j].vars[0]) <= 0.1) {
#endif	
		OutputPoint(pf2dOutfile, &(avctLines[NumOfPoints*i + j]), 2);
		EvaluateParEqn(apeqnSurface, &(avctLines[NumOfPoints*i + j]), 
                  &vctTemp);
		OutputPoint(pf3dOutfile, &vctTemp, 3);

#ifdef MOEBIUS
	    }
#endif
	}
    }
}



/*
 *   Function: cdpgu1
 *
 *   Author:   Jim Lambers
 */

SCALAR	cdpgu1(index, pvctL)
short	index;
PVECTOR	pvctL;
{
    if (!index) 
        vctGlobalOldCDP = *(ParCurv(1, pvctL, &vctGlobalOldCD));

    return(vctGlobalOldCDP.vars[index]);
}



/*
 *   Function: cdpgu2
 *
 *   Author:   Jim Lambers
 */

SCALAR	cdpgu2(index, pvctL)
short	index;
PVECTOR	pvctL;
{
    if (!index) 
        vctGlobalOldCDP = *(ParCurv(2, pvctL, &vctGlobalOldCD));

    return(vctGlobalOldCDP.vars[index]);
}



/*
 *   Function: ParCurv
 *
 *   Author:   Jim Lambers
 */

PVECTOR	ParCurv(value, pvctPoint1, pvctCurved)
short	value;
PVECTOR	pvctPoint1, pvctCurved;
{
    short	i = 0, j;
    VECTOR	vctTemp1, vctTemp2, vctXTemp[3], vctNBar, vctBigN;
    SCALAR	ascGs[2][2], scG, ascL[3], scBigH, scBigL, scGaussK, scKmax, 
                  scKmin, scKval, scStarAlpha1, scStarAlpha2, scTildeAlpha1, 
                  scTildeAlpha2, scBarAlpha1, scBarAlpha2, scAlpha1, scAlpha2, 
                  scLE, scBarLength, scBigLE, scAlpha1Old = 0.0, 
                  scAlpha2Old = 0.0;
    PVECTOR	pvctDirection;


    pvctDirection = (PVECTOR) MyMalloc(sizeof(VECTOR));

    for (i = 0; i < 2; i++)
	for (j = 0; j < 2; j++)
	    ascGs[i][j] = EvaluateEqn(apeqnGsMatrix[i][j], pvctPoint1);
    
    scG = (ascGs[0][0] * ascGs[1][1]) - (ascGs[0][1] * ascGs[0][1]);
    if (fabs(scG) < 0.000001)
        fprintf(stderr, "Surface is degenerate at this point\n");

    for (i = 0; i < NUMBER_OF_VARS; i++) {
	EvaluateParEqn(apeqnXSVVF[0], pvctPoint1, &vctTemp1); 
	EvaluateParEqn(apeqnXSVVF[1], pvctPoint1, &vctTemp2); 
    }

    VctCrossproduct(&vctTemp1, &vctTemp2, &vctNBar);

    scBarLength = VctDotVct(&vctNBar, &vctNBar);

    /* Avoid core dumps on surfaces that would generate them */
    if (scBarLength == 0.0)
        FatalExit("scG is zero in ParCurv", FALSE);

    VctTimesSc(&vctNBar, sqrt(1.0 / scBarLength), &vctBigN);
    
    EvaluateParEqn(apeqnX00, pvctPoint1, &(vctXTemp[0]));
    EvaluateParEqn(apeqnX01, pvctPoint1, &(vctXTemp[1]));
    EvaluateParEqn(apeqnX11, pvctPoint1, &(vctXTemp[2]));

    for (i = 0; i < NUMBER_OF_VARS; i++)
	ascL[i] = VctDotVct(&vctBigN, &(vctXTemp[i]));

    scBigL = (ascL[0] * ascL[2]) - (ascL[1] * ascL[1]);

    /* Avoid core dumps on surfaces that would generate them */
    if (scG == 0.0)
        FatalExit("scG is zero in ParCurv", FALSE);

    scGaussK = scBigL / scG;   

    scBigH = ((ascGs[0][0] * ascL[2]) + (- (2.0 * ascGs[0][1] * ascL[1])) + 
      (ascGs[1][1] * ascL[0])) / (2 * scG);

    scKmax = scBigH + sqrt((scBigH * scBigH) - scGaussK);

    scKmin = scBigH - sqrt((scBigH * scBigH) - scGaussK);

    if ( fabs(scKmax - scKmin) < 0.000001 )
	fprintf(stderr, "Almost an umbilic in pcp: %.10lf\n", scKmin -  scKmax);

    if (value == 1)
        scKval = scKmax;
    else scKval = scKmin;
    
    scStarAlpha1 = (scKval * ascGs[0][1]) - ascL[1];
    scStarAlpha2 = -(scKval * ascGs[0][0]) + ascL[0];
    scTildeAlpha1 = (scKval * ascGs[1][1]) - ascL[2];
    scTildeAlpha2 = -(scKval * ascGs[1][0]) + ascL[1];

    if ( (fabs(scStarAlpha1) + fabs(scStarAlpha2)) > (fabs(scTildeAlpha1) + 
      fabs(scTildeAlpha1)) ) {
	scBarAlpha1 = scStarAlpha1;
	scBarAlpha2 = scStarAlpha2;
    }
    else {
	scBarAlpha1 = scTildeAlpha1;
	scBarAlpha2 = scTildeAlpha2;
    }

    scLE = sqrt((scBarAlpha1 * scBarAlpha1 * ascGs[0][0]) + (2 * scBarAlpha1 * 
      scBarAlpha2 * ascGs[0][1]) + (scBarAlpha2 * scBarAlpha2 * 
      ascGs[1][1]));
    
    if ((fabs(scBarAlpha1) + fabs(scBarAlpha2)) < 0.000001) {
	if (!(IsZeroVct(&vctGlobalOldCDP))) 
	    return(&vctGlobalOldCDP);
	else if (!(IsZeroVct(pvctCurved))) {
	    fprintf(stderr, "Using supplied curvature direction\n");
	    return(pvctCurved);
	}
    }
    else {
	if (fabs(scLE) < 0.000001) {
	    if (!(IsZeroVct(&vctGlobalOldCDP)))
		return(&vctGlobalOldCDP);
	    else if (!(IsZeroVct(pvctCurved))) {
		fprintf(stderr, "Using supplied curvature direction\n");
		return(pvctCurved);
	    }
	}
    }
    scBigLE = scLE;

    scAlpha1 = scBarAlpha1 / scBigLE;
    scAlpha2 = scBarAlpha2 / scBigLE;

    if ((fabs(-(scAlpha1) - scAlpha1Old) + fabs(-(scAlpha2) - scAlpha2Old)) <
	(fabs(scAlpha1 - scAlpha1Old) + fabs(scAlpha2 - scAlpha2Old))) {
	scAlpha1 = -scAlpha1;
	scAlpha2 = -scAlpha2;
    }

    scAlpha1Old = scAlpha1;
    scAlpha2Old = scAlpha2;
	      
    pvctDirection->vars[0] = scAlpha1;
    pvctDirection->vars[1] = scAlpha2;
    pvctDirection->vars[2] = 0.0;
    return(pvctDirection);
}

