/*
 *  Module:   polynomial parse
 *
 *  Library:  mathbase.a (project Riemann math routines)
 */

#include "parsepoly.h"
#include "ctype.h"


static int    TokenType;                /* Type of current token */
static SCALAR scTokenValue;             /* Current token; all numbers are */
                                        /*   read in as SCALAR's; they can */
                                        /*   be cast to int's if desired */
static char   *sGlobalPoly;
static int     current;                 /* Current position in string */


/*
 *   Function:	ParsePoly
 *
 *   Purpose:	This function parses the passed string (assumed to contain a 
 *		polynomial).  Valid polynomials contain no times signs.  This
 *		function mallocs space for the polynomial and its terms.  This
 *		function returns TRUE if the passed poly is valid and FALSE 
 *		otherwise.
 *
 *   Author:    Bret Johnson
 *
 */

PPOLY ParsePoly(sPoly)
char	*sPoly;
{
    PTERM	ptrm;			/* Pointer to current term */
    BOOL	bFirstTerm = TRUE;
    BOOL	bResult;
    PPOLY	ppl;


    current = 0;
    ppl = (PPOLY) MyMalloc(sizeof(POLY));

    sGlobalPoly = sPoly;

    if (!GetNextToken()) 		/* Get the first token */
	return NULL;

    if (TokenType == END_TYPE)	/* Null polynomials are invalid */
	return NULL;

    /* Parse each term */
    while (TokenType != END_TYPE) {

	/* Parse '+' or '-' sign and the following term */
	if (bFirstTerm) {
	    ppl->ptrmFirst = (PTERM) MyMalloc( sizeof(TERM) );
	    ptrm = ppl->ptrmFirst;
	    bFirstTerm = FALSE;

	    bResult = ParseTerm(ptrm);
	}
	else {
	    ptrm->ptrmNext = (PTERM) MyMalloc( sizeof(TERM) );
	    ptrm = ptrm->ptrmNext;

	    if (TokenType == MINUS_TYPE) {
    		if (!GetNextToken())		/* Scan past minus */
		    return NULL;
		bResult = ParseTerm(ptrm);
		ptrm->coef *= -1.0;
	    }
	    else if (TokenType == PLUS_TYPE) {
    		if (!GetNextToken())		/* Scan past plus */
		    return NULL;
		bResult = ParseTerm(ptrm);
	    }
	    else return NULL;		/* Bad polynomial */

	} /* if (bFirstTerm) else */

 	if (!bResult)			/* If ParseTerm detected a bad poly, */
	    return NULL;		/*   return FALSE */

    } /* while */

    return ppl;
}



/*
 *   Function:	ParseTerm
 *
 *   Purpose:	This function parses a polynomial term.  It returns TRUE if
 *		it finds no errors in the polynomial term.
 *
 *   Author:    Bret Johnson
 *
 */

BOOL ParseTerm(ptrm)
PTERM	ptrm;
{
    int	    i;


    /* Clear term to default */
    for (i = 0; i < NUMBER_OF_VARS; ++i)
        ptrm->exps[i] = 0;
    
    if ( !ParseCoefficient(ptrm) )
	return FALSE;

    /* Parse the rest of the term */
    while (TokenType == X_VAR_TYPE || TokenType == Y_VAR_TYPE || 
      TokenType == Z_VAR_TYPE)
 
	if ( !ParseVariable(ptrm) )
	    return FALSE;

    return TRUE;
}



/*
 *   Function:	ParseCoefficient
 *
 *   Purpose:	This function parses a polynomial term coefficient.  It returns
 *		TRUE if it finds no errors in the coefficient.
 *
 *   Author:    Bret Johnson
 *
 */

BOOL ParseCoefficient(ptrm)
PTERM	ptrm;
{
    /* Take care of a possible leading minus sign */
    if (TokenType == MINUS_TYPE) {
	ptrm->coef = -1.0;
        if (!GetNextToken())
	    return FALSE;
    }
    else ptrm->coef = 1.0;

    /* If the coefficient is implied to be plus or minus one, check that */
    /*   a variable follows the implied coefficient */
    if (TokenType != NUMBER_TYPE)
	return (TokenType == X_VAR_TYPE || TokenType == Y_VAR_TYPE || 
	  TokenType == Z_VAR_TYPE);

    ptrm->coef *= scTokenValue; 	   /* Process the numeric coefficient */
    return GetNextToken();
}



/*
 *   Function:	ParseVariable
 *
 *   Purpose:	This function parses a single variable (and its associated
 *		exponent) in a polynomial term.  This function returns TRUE if
 *		it finds no errors.
 *
 *   Author:    Bret Johnson
 *
 */

BOOL ParseVariable(ptrm)
PTERM	ptrm;
{
    int		variable;	/* Variable being processed: X, Y, or Z */


    variable = TokenType;	        /* Record variable */

    if (ptrm->exps[variable] != 0)	/* Don't permit same variable */
	return FALSE;			/*   occuring more than once in term */

    ptrm->exps[variable] = 1;		/* Default exponent is one */
    if (!GetNextToken())
	return FALSE;

    if (TokenType != CARET_TYPE)
	return TRUE;

    if (!GetNextToken()) 		/* Scan past caret */
	return FALSE;

    if (TokenType == MINUS_TYPE) {
	ptrm->exps[variable] = -1;
        if (!GetNextToken())
	    return FALSE;
    }

    if (TokenType != NUMBER_TYPE)
	return FALSE;

    ptrm->exps[variable] *= (EXPONENT) scTokenValue;

    return GetNextToken();		/* Scan past exponent */
}



/*
 *   Function:	GetNextToken
 *
 *   Purpose:	This function is the lexical analyser part of the polynomial 
 *		parser.  It sets the TokenType and TokenValue global variables
 *		to the the next token in the polynomial string.
 *
 *   Author:    Bret Johnson
 *
 */

BOOL GetNextToken()
{
    char   c;				    /* Character at current position */


    c = sGlobalPoly[current];

    if (c == '\0')
	TokenType = END_TYPE;
    else if (isspace(c)) {
	++current;
	GetNextToken();			/* Skip spaces via recursion */
    }
    else if (c == '+') {
	TokenType = PLUS_TYPE;
	++current;
    }
    else if (c == '-') {
	TokenType = MINUS_TYPE;
	++current;
    }
    else if (c == '^') {
	TokenType = CARET_TYPE;
	++current;
    }
    else if (c == 'x' || c == 's') {	/* 'x' and 's' are synonyms, as are */
	TokenType = X_VAR_TYPE;		/*   'y' and 't';  'x' and 'y' are */
	++current;			/*   meant to be used in implicit */
    }					/*   equations while 's' and 't' */
    else if (c == 'y' || c == 't') {	/*   are meant to be used in */
	TokenType = Y_VAR_TYPE;		/*   parametric equations */
	++current;
    }
    else if (c == 'z') {
	TokenType = Z_VAR_TYPE;	
	++current;
    }
    else if (isdigit(c) || c == '.') {

	TokenType = NUMBER_TYPE;
	sscanf(&(sGlobalPoly[current]), "%lf", &scTokenValue);

	/* Skip to next token */
	while (isdigit(c) || c == '.') {
	    ++current;
	    c = sGlobalPoly[current];
	}
    }
    else return FALSE;			/* Illegal character */

    return TRUE;
}

