/*
 *  Module:  numerical quick
 *
 *  Program: numgeod (numerical geodesics generator for project Riemann)
 */

#include "master.h"

/*
A quick approximation algorithm for geodesics of an
implicitly defined surface

Given: an initial point p(n) and a tangent vector v(n).

f'(x) is the total derivative of the surface at x (f'(x) = (grad f)(x))

p(n)~ = p(n) + (v(n) * step)

		< v(n), f'(p(n)~) >
proj(v(n)) = ------------------------ f'(p(n)~)
	     < f'(p(n)~), f'(p(n)~) >

refl(v(n)) = v(n) - proj(v(n))

refl(v(n)) is the reflection of v(n) at the plane normal to f'(p(n)~).
  _
p(n) = p(n)~ + (refl(v(n)) * step)

The resulting point and tangent vector:

	   _            f(p(n))             _
p(n+1) = p(n) - -------_---------_---- f'(p(n))
	        < f'(p(n)), f'(p(n)) >
		  _ 
	      < v(n), f'(p(n+1)) >
           -------------------------- f'(p(n+1))
	   < f'(p(n+1)), f'(p(n+1)) >
v(n+1) = -----------------------------------------
	 |   < v(n), f'(p(n+1)) >                |
         | -------------------------- f'(p(n+1)) |
	 | < f'(p(n+1)), f'(p(n+1)) >            |
*/

/*
 *   Function:	NumGradient
 *
 *   Purpose:	This function returns the gradient of Eqn evaluated at the
 *		point Pt.
 *
 *   Author:    Jim Lambers
 */

void	NumGradient(peqn, pvctPt, pvctGradient)
     PEQN	peqn;
     PVECTOR	pvctPt, pvctGradient;
{
    short	i, j;
    SCALAR	h = 0.001;
    PVECTOR	pvctH1[NUMBER_OF_VARS];
    PVECTOR	pvctH2[NUMBER_OF_VARS];


    for (j = 0; j < NUMBER_OF_VARS; j++) {
	pvctH1[j] = (PVECTOR) MyMalloc(sizeof(VECTOR)); 
	pvctH2[j] = (PVECTOR) MyMalloc(sizeof(VECTOR)); 
	for (i = 0; (i < NUMBER_OF_VARS); i++) {
	    pvctH1[j]->vars[i] = pvctPt->vars[i];
	    pvctH2[j]->vars[i] = pvctPt->vars[i];
	}
    } 
    for (i = 0; i < NUMBER_OF_VARS; i++) {
	pvctH1[i]->vars[i] += h;
	pvctH2[i]->vars[i] -= h;
    }
    for (i = 0; i < NUMBER_OF_VARS; i++) {
	pvctGradient->vars[i] = (EvaluateEqn(peqn, pvctH1[i]) - 
			     EvaluateEqn(peqn, pvctH2[i])) / (2 * h);
    }
}



/*
 *   Function:	NumQuick
 *
 *   Purpose:	This function, given an initial point, initial vector, and 
 *		equation, computes approximations to geodesics.
 *
 *   Author:    Jim Lambers
 *
 */

void 	NumQuick(pvctPoint, pvct, scStep, iterations, pfOutfile, peqn)
     PVECTOR pvctPoint, pvct;
     SCALAR  scStep;
     short   iterations;
     FILE    *pfOutfile;
     PEQN    peqn;
{
    VECTOR 	vctPn, vctVn, vctPnTilde, vctPnBar, vctGradTilde,
                vctVnBar, vctGradBar, vctGradPn, vctRefl, vctTemp;
    SCALAR 	scBetaBar;
    short	j;


    VctCopy(&vctPn, pvctPoint);
    VctCopy(&vctVn, pvct);

    for (j = 0; j <= iterations; j++) {
	VctTimesSc(&vctVn, scStep, &vctTemp);
	VctPlusVct(&vctPn, &vctTemp, &vctPnTilde);
        OutputPoint(pfOutfile, &vctPn, 3);
	NumGradient(peqn, &vctPnTilde, &vctGradTilde);
	Reflection(&vctVn, &vctGradTilde, &vctVnBar);
	VctTimesSc(&vctVnBar, scStep, &vctTemp);
	VctPlusVct(&vctPnTilde, &vctTemp, &vctPnBar);
	NumGradient(peqn, &vctPnBar, &vctGradBar);
	scBetaBar = - (EvaluateEqn(peqn, &vctPnBar) / 
          VctDotVct(&vctGradBar, &vctGradBar));
	VctTimesSc(&vctGradBar, scBetaBar, &vctTemp);
	VctPlusVct(&vctPnBar, &vctTemp, &vctPn);
	NumGradient(peqn, &vctPn, &vctGradPn);
	Reflection(&vctVnBar, &vctGradPn, &vctRefl);
	Normalize(&vctRefl, &vctVn);
    }
}

