/*
*  Module:   output
*
*  Programs: symgeod, numgeod, impcurv, and parcurv (project Riemann curvature
*            lines and geodesics generators)
*/

#include <sys/time.h>
#include <sys/types.h>
#include "master.h"

time_t	time();

/*
 *   Function:	OutputHeader
 *
 *   Purpose:	This function outputs a descriptive file header to the passed 
 *              output file.  This routine should be called at the beginning of
 *              the math programs to generate a header which can be read by 
 *		xdisp, the X11 project Riemann graphics displayer.  type 
 *		describes the type of data in the file.  It must be IMP_GEOD, 
 *		IMP_CURV, PAR_GEOD, or PAR_CURV.  sTitle is the desired title 
 *		(typically what the user inputs).  et is the equation type of 
 *		the equation(s) in asEqn.  asEqn should be a pointer to a string
 *		for the implicit case (only one equation) and should be an 
 *		array of three strings in the parametric case.  b2dOutput should
 *		be TRUE if this file will contain parameter space output.
 *
 *   Author:    Bret Johnson
 */

void	OutputHeader(pf, type, sTitle, et, asEqn, b2dOutput)
FILE    *pf;
int	type;
char	*sTitle;
EQNTYPE	et;
char	*asEqn[];
BOOL	b2dOutput;
{
    time_t	clock;			/* Current date and time */
    int		i;


    /* Truncate all strings to MAX_STRING_LENGTH characters */

    if (strlen(sTitle) >= MAX_STRING_LENGTH) 
	sTitle[MAX_STRING_LENGTH-1] = '\0';

    if (strlen(asEqn[0]) >= MAX_STRING_LENGTH)
	asEqn[0][MAX_STRING_LENGTH-1] = '\0';

    if (IS_PARAMETRIC(type)) {
        if (strlen(asEqn[1]) >= MAX_STRING_LENGTH)
	    asEqn[1][MAX_STRING_LENGTH-1] = '\0';
        if (strlen(asEqn[2]) >= MAX_STRING_LENGTH)
	    asEqn[2][MAX_STRING_LENGTH-1] = '\0';
    }

    switch (type) {
	case IMP_GEOD:
	    fprintf(pf, "Type: implicit geodesics\n");
	    break;
	case IMP_CURV:
	    fprintf(pf, "Type: implicit curvature lines\n");
	    break;
	case PAR_GEOD:
	    fprintf(pf, "Type: parametric geodesics\n");
	    break;
	case PAR_CURV:
	    fprintf(pf, "Type: parametric curvature lines\n");
	    break;
    }

    /* Output the type of the space for the parametric types */
    if (IS_PARAMETRIC(type))
	fprintf(pf, "Space: %s\n", (b2dOutput ? "parametric" : "normal"));

    fprintf(pf, "Title: %s\n", sTitle);

    /* Output date and time */
   // clock = time((time_t *) 0);
    fprintf(pf, "Created: 26.06.2010\n");

    /* Output the equation type */
    if (et == E_POLY)
        fprintf(pf, "Equation type: polynomial\n");
    else fprintf(pf, "Equation type: expression\n");

    if (IS_IMPLICIT(type))
        fprintf(pf, "Implicit equation: %s = 0\n", asEqn[0]);
    else
	for (i = 0; i < 3; i++)
	    fprintf(pf, "Parametric equation %d: %c = %s\n", i + 1, 
	      i + 'x', asEqn[i]);
}



/*
 *   Function:	OutputLineStart
 *
 *   Purpose:	This function outputs to file pf a header describing the start 
 *		of a new line.  type describes the type of data in the file.
 *              It must be IMP_GEOD, IMP_CURV, PAR_GEOD, or PAR_CURV.  direction
 *              is the direction of curvature lines (either 1 or 2).
 *
 *   Author:    Bret Johnson
 */

void	OutputLineStart(pf, type, direction)
FILE	*pf;
int	type,
	direction;
{
    if (IS_GEODESICS(type))
        fprintf(pf, "Line:\n");
    else fprintf(pf, "Line in direction %d:\n", direction);
}



/*
 *   Function:	OutputPoint
 *
 *   Purpose:	This function outputs the passed point to the passed file.
 *		dimensions is the number of dimensions associatied with the 
 *		point.  It should be 2 when dealing with the parameter space and
 *		3 otherwise.
 *
 *   Author:    Jim Lambers
 */

void OutputPoint(pf, pvct, dimensions)
FILE   	*pf;
PVECTOR	pvct;
int	dimensions;
{
    int		i;
   
    for (i = 0; i < dimensions; i++) {
        fprintf(pf, "%lf", pvct->vars[i]);
	fputc( (i < dimensions - 1 ? ' ' : '\n'), pf );
    }
}

