/*
 *  Module:  quick
 *
 *  Program: symgeod (symbolic geodesics generator for project Riemann)
 */

#include "geodesics.h"

/*
A quick approximation algorithm for geodesics of an implicitly defined surface:

Given: an initial point p(n) and a tangent vector v(n).

f'(x) is the total derivative of the surface at x (f'(x) = (grad f)(x))

p(n)~ = p(n) + (v(n) * step)

		< v(n), f'(p(n)~) >
proj(v(n)) = ------------------------ f'(p(n)~)
	     < f'(p(n)~), f'(p(n)~) >

refl(v(n)) = v(n) - proj(v(n))

refl(v(n)) is the reflection of v(n) at the plane normal to f'(p(n)~).
  _
p(n) = p(n)~ + (refl(v(n)) * step)

The resulting point and tangent vector:

	   _            f(p(n))             _
p(n+1) = p(n) - -------_---------_---- f'(p(n))
	        < f'(p(n)), f'(p(n)) >
		  _ 
	      < v(n), f'(p(n+1)) >
           -------------------------- f'(p(n+1))
	   < f'(p(n+1)), f'(p(n+1)) >
v(n+1) = -----------------------------------------
	 |   < v(n), f'(p(n+1)) >                |
         | -------------------------- f'(p(n+1)) |
	 | < f'(p(n+1)), f'(p(n+1)) >            |
*/

/*
 *   Function:	Quick
 *
 *   Purpose:	This function, given an initial point, initial vector, and 
 *		equation, computes approximations to geodesics.
 *
 *   Author:    Jim Lambers
 *
 */

void 	quick(peqn, apeqnPartials, pvctPoint, pvct, scStep, iterations, 
  pfOutfile)
PEQN	peqn, apeqnPartials[NUMBER_OF_VARS];
PVECTOR pvctPoint, pvct;
SCALAR  scStep;
int	iterations;
FILE	*pfOutfile;
{
    VECTOR 	vctPn, vctVn, vctPnTilde, vctPnBar, vctGradTilde, vctVnBar, 
		  vctGradBar, vctGradPn, vctRefl, vctTemp;
    SCALAR 	scBetaBar;
    int		j;


    VctCopy(&vctPn, pvctPoint);
    VctCopy(&vctVn, pvct);

    for (j = 0; j <= iterations; j++) {
	VctTimesSc(&vctVn, scStep, &vctTemp);
	VctPlusVct(&vctPn, &vctTemp, &vctPnTilde);
	OutputPoint(pfOutfile, &vctPn, 3);
	EqnGradient(apeqnPartials, &vctPnTilde, &vctGradTilde);
	Reflection(&vctVn, &vctGradTilde, &vctVnBar);
	VctTimesSc(&vctVnBar, scStep, &vctTemp);
	VctPlusVct(&vctPnTilde, &vctTemp, &vctPnBar);
	EqnGradient(apeqnPartials, &vctPnBar, &vctGradBar);
	scBetaBar = - (EvaluateEqn(peqn, &vctPnBar) / 
                      VctDotVct(&vctGradBar, &vctGradBar));
	VctTimesSc(&vctGradBar, scBetaBar, &vctTemp);
	VctPlusVct(&vctPnBar, &vctTemp, &vctPn);
	EqnGradient(apeqnPartials, &vctPn, &vctGradPn);
	Reflection(&vctVnBar, &vctGradPn, &vctRefl);
	Normalize(&vctRefl, &vctVn);
    }
}

