/*
*  Module:  implicit curvature lines user interface
*
*  Program: impcurv (implicit curvature lines generator for project Riemann)
*/

#define MAIN			/* Define global variables in this file */
#include "userinter.h"

/*
 *   Function:	UserInterface
 *
 *   Purpose:	This function handles the user interface for implicit curvature
 *		lines.  It scans the command line for the appropriate parameters
 *		or prompts the user for them if the user didn't enter any 
 *		command line parameters.  
 *
 *   Author:    Bret Johnson
 */

void UserInterface(argc, argv, pvctInitial, pscStep, pscSpacing, pscD1, pscD2,
  peqn, ppfOutput, psTitle, psEqn)
int 	argc;
char 	*argv[];
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR	*pscSpacing;
SCALAR	*pscD1, *pscD2;
PEQN	peqn;
FILE	**ppfOutput;
char	**psTitle;
char	**psEqn;
{
    int	i;


    /* Set up default parameters */
    for (i = 0; i < NUMBER_OF_VARS; ++i)
        pvctInitial->vars[i] = 0.0;
    *pscStep = DEF_CURV_STEP;
    *pscSpacing = DEF_CURV_SPACING;
    *pscD1 = DEF_CURV_D1;
    *pscD2 = DEF_CURV_D2;
    *ppfOutput = stdout;
    peqn->et = DEF_EQNTYPE;
    *psTitle = "";

    GlobalArgv = argv;
    GlobalArgc = argc;

    /* Form the usage error message */
    sprintf(sUsage, "Usage: %s [-t title] [-v initial_vector] [-s step_size]\n\
  [-sp mesh_spacing] [-1 direction_one_distance] [-2 direction_two_distance]\n\
  [output_file] -e expression | -p polynomial", GlobalArgv[0]);

    /* If the user entered no command line arguments, assume he wishes to */
    /*   use the interactive interface.  Otherwise, assume he is using */
    /*   the command line interface. */

    if (argc != 1) {

        /* Process the command line arguments */
        ProcessArgs(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, peqn, 
          ppfOutput, psTitle, psEqn);

        /* Check that the initial point is on the surface */
        if (fabs(EvaluateEqn(peqn, pvctInitial)) > EPSILON) 
            FatalExit("The initial point is not on the surface", FALSE);
    }
    else InteractiveInput(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, peqn,
           ppfOutput, psTitle, psEqn);

} /* UserInterface */



/*
 *   Function:	ProcessArgs
 *
 *   Purpose:	This function processes all of the command line arguments.
 *
 *   Author:    Bret Johnson
 */

void ProcessArgs(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, peqn, 
  ppfOutput, psTitle, psEqn)
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR  *pscSpacing; 
SCALAR	*pscD1, *pscD2;
PEQN    peqn;
FILE    **ppfOutput;
char    **psTitle, **psEqn;
{
    BOOL	bInit = FALSE,	     /* TRUE when initial point entered */
    		bStep = FALSE,	     /* TRUE when step size entered */
                bSpacing = FALSE,    /* TRUE when mesh spacing entered */
                bD1 = FALSE,         /* TRUE when direction one and two */
                bD2 = FALSE,         /*   distances entered */
          	bOutput	= FALSE,     /* TRUE when output file entered */
    		bEqn = FALSE;        /* TRUE when user inputs the equation */
    char	*sArg;		     /* Current argument string */
    int		i;		     /* Argument counter */
    char	sBuffer[100];


    for (i = 1; i < GlobalArgc; ++i) {

        sArg = GlobalArgv[i];

        if (strcmp(sArg, "-t") == 0) {
	    ++i;
	    if (i >= GlobalArgc)
	        FatalExit("Failed to specify a title", TRUE);
	    *psTitle = GlobalArgv[i];
	}
	else if (strcmp(sArg, "-v") == 0) {
	    GetPointArg(3, pvctInitial, &i);	
	    bInit = TRUE;
	}
	else if (strcmp(sArg, "-s") == 0) {
	    GetStepSizeArg(pscStep, &i);
	    bStep = TRUE;
	}
	else if (strcmp(sArg, "-sp") == 0) {
            GetSpacingArg(pscSpacing, &i);
            bSpacing = TRUE;
	}
	else if (strcmp(sArg, "-1") == 0) {
            GetD1Arg(pscD1, &i);
            bD1 = TRUE;
	}
	else if (strcmp(sArg, "-2") == 0) {
            GetD2Arg(pscD2, &i);
            bD2 = TRUE;
	}
	else if (strcmp(sArg, "-e") == 0) {
	    GetExprArg(peqn, psEqn, &i);
	    bEqn = TRUE;
	}
	else if (strcmp(sArg, "-p") == 0) {
	    GetPolyArg(peqn, psEqn, &i);
	    bEqn = TRUE;
	}
	else  if (sArg[0] == '-' || bOutput) {
            sprintf(sBuffer, "Illegal option: \"%s\"", sArg);
	    FatalExit(sBuffer, TRUE);
        }
	else {
            *ppfOutput = fopen(sArg, "w");

	    if (*ppfOutput == NULL)
 	        FatalExit("Error opening specified file for output", FALSE);

	    bOutput = TRUE;
	}
    } /* for */

    if (!bEqn)
	FatalExit("Failed to specify an expression or polynomial", TRUE);
    if (!bInit)
        fprintf(stderr, "Assuming initial point is (0.0, 0.0, 0.0)\n");
    if (!bStep)
        fprintf(stderr, "Assuming step size is %.2f\n", DEF_CURV_STEP);
    if (!bSpacing)
        fprintf(stderr, "Assuming mesh spacing is %.1f\n", DEF_CURV_SPACING);
    if (!bD1)
        fprintf(stderr, "Assuming direction one distance is %.1f\n",
          DEF_CURV_D1);
    if (!bD2)
        fprintf(stderr, "Assuming direction two distance is %.1f\n",
          DEF_CURV_D2);
    if (!bOutput)
        fprintf(stderr, "Assuming output goes to standard output\n");
    fprintf(stderr, "\n");

} /* ProcessArgs */



/*
 *   Function:	InteractiveInput
 *
 *   Purpose:	This function handles the interactive part of the user 
 *              interface for implicit curvature lines.  It prompts the user 
 *		for the user-specified parameters and inputs these parameters.
 *
 *   Author:    Bret Johnson
 */

void InteractiveInput(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, peqn,
  ppfOutput, psTitle, psEqn)
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR  *pscSpacing;
SCALAR  *pscD1, *pscD2;
PEQN    peqn;
FILE    **ppfOutput;
char    **psEqn, **psTitle;
{
    char	sBuffer[MAX_STRING_LENGTH];	     /* String buffer */


    *psTitle = InputTitle(); 		/* Input the title */

    /* Input the initial point */
    Input("X coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[0]), "%lf");
    Input("Y coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[1]), "%lf");
    Input("Z coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[2]), "%lf");

    /* Input the step size */
    sprintf(sBuffer, "Step size [%.2f]: ", DEF_CURV_STEP);
    Input(sBuffer, (char *) pscStep, "%lf");

    /* Input the mesh spacing */
    sprintf(sBuffer, "Mesh spacing [%.1f]: ", DEF_CURV_SPACING);
    Input(sBuffer, (char *) pscSpacing, "%lf");

    /* Input the direction one distance */
    sprintf(sBuffer, "Distance along direction one [%.1f]: ", DEF_CURV_D1);
    Input(sBuffer, (char *) pscD1, "%lf");

    /* Input the direction two distance */
    sprintf(sBuffer, "Distance along direction two [%.1f]: ", DEF_CURV_D2);
    Input(sBuffer, (char *) pscD2, "%lf");

    /* Input the equation type */
    peqn->et = InputEqnType();

    /* Input the expression or polynomial */
    if (peqn->et == E_EXPR)
        peqn->d.pexp = InputExpr(psEqn, '\0');
    else
        peqn->d.ppl = InputPoly(psEqn, '\0');

    /* Check that the initial point is on the surface */
    if (fabs(EvaluateEqn(peqn, pvctInitial)) > EPSILON)
        FatalExit("The initial point is not on the surface", FALSE);

    /* Input the file name where output is to go */
    *ppfOutput = InputOutputFile(
      "Name of file to hold output [standard output]: ", stdout);
}

