/*
*  Module:   implicit geodesics lines user interface
*
*  Programs: symgeod & numgeod (sybolic and numerical implicit geodesics 
*            generators for project Riemann)
*/

#define MAIN			/* Define global variables with this file */
#include "userinter.h"


/*
 *   Function:	UserInterface
 *
 *   Purpose:	This function handles the user interface for implicit geodesics.
 *		It scans the command line for the appropriate parameters or 
 *		prompts the user for them if the user didn't enter any command 
 *		line parameters.
 *
 *   Author:    Bret Johnson
 */

void UserInterface(argc, argv, pvctInitial, pscStep, pIterations, pGeodesics, 
  peqn, ppfOutput, psTitle, psEqn)
int 	argc;
char 	*argv[];
PVECTOR pvctInitial;
SCALAR  *pscStep;
short	*pIterations;
short	*pGeodesics;
PEQN	peqn;
FILE	**ppfOutput;
char	**psTitle;
char	**psEqn;
{
    int	i;


    /* Set up default parameters */
    for (i = 0; i < NUMBER_OF_VARS; ++i)
        pvctInitial->vars[i] = 0.0;
    *pscStep = DEF_GEOD_STEP;
    *pIterations = DEF_GEOD_ITERATIONS;
    *pGeodesics = DEF_GEODESICS;
    peqn->et = DEF_EQNTYPE;
    *ppfOutput = stdout;
    *psTitle = "";

    GlobalArgv = argv;
    GlobalArgc = argc;

    /* Form the usage error message */

    sprintf(sUsage,"Usage: %s [-t title] [-v initial_vector] [-s step_size] \n\
  [-i iterations_per_geodesic] [-g geodesics] [output_file] -e expression | \n\
  -p polynomial", GlobalArgv[0]);

    /* If the user entered no command line arguments, assume he wishes to */
    /*   use the interactive interface.  Otherwise, assume he is using */
    /*   the command line interface. */

    if (argc != 1) {

        /* Process the command line arguments */
        ProcessArgs(pvctInitial, pscStep, pIterations, pGeodesics, peqn, 
	  ppfOutput, psTitle, psEqn);

        /* Check that the initial point is on the surface */
        if (fabs(EvaluateEqn(peqn, pvctInitial)) > EPSILON) 
            FatalExit("The initial point is not on the surface", FALSE);
    }
    else InteractiveInput(pvctInitial, pscStep, pIterations, pGeodesics, peqn, 
	   ppfOutput, psTitle, psEqn);

} /* UserInterface */


/*
 *   Function:	ProcessArgs
 *
 *   Purpose:	This function processes all of the command line arguments.
 *
 *   Author:    Bret Johnson
 */

void ProcessArgs(pvctInitial, pscStep, pIterations, pGeodesics, peqn, ppfOutput,
  psTitle, psEqn)
PVECTOR pvctInitial;
SCALAR	*pscStep;
short	*pIterations;
short	*pGeodesics;
PEQN	peqn;
FILE	**ppfOutput;
char	**psTitle;
char	**psEqn;
{
    BOOL	bInit = FALSE;	     /* TRUE when initial point entered */
    BOOL	bStep = FALSE;	     /* TRUE when step size entered */
    BOOL	bIterations = FALSE; /* TRUE when num. of iterations entered */
    BOOL	bGeodesics = FALSE;  /* TRUE when number of geodesics entered */
    BOOL	bOutput	= FALSE;     /* TRUE when output file entered */
    BOOL	bEqn = FALSE;        /* TRUE when user inputs the equation */
    char	*sArg;		     /* Current argument string */
    int		i;		     /* Argument counter */
    char	sBuffer[100];


    for (i = 1; i < GlobalArgc; ++i) {

        sArg = GlobalArgv[i];

        if (strcmp(sArg, "-t") == 0) {
	    ++i;
	    if (i >= GlobalArgc)
	        FatalExit("Failed to specify a title", TRUE);
	    *psTitle = GlobalArgv[i];
	}
	else if (strcmp(sArg, "-v") == 0) {
	    GetPointArg(3, pvctInitial, &i);	
	    bInit = TRUE;
	}
	else if (strcmp(sArg, "-s") == 0) {
	    GetStepSizeArg(pscStep, &i);
	    bStep = TRUE;
	}
	else if (strcmp(sArg, "-i") == 0) {
	    GetIterationsArg(pIterations, &i);
	    bIterations = TRUE;
	}
	else if (strcmp(sArg, "-g") == 0) {
	    GetGeodesicsArg(pGeodesics, &i);
	    bGeodesics = TRUE;
	}
	else if (strcmp(sArg, "-e") == 0) {
	    GetExprArg(peqn, psEqn, &i);
	    bEqn = TRUE;
	}
	else if (strcmp(sArg, "-p") == 0) {
	    GetPolyArg(peqn, psEqn, &i);
	    bEqn = TRUE;
	}
	else  if (sArg[0] == '-' || bOutput) {
            sprintf(sBuffer, "Illegal option: \"%s\"", sArg);
	    FatalExit(sBuffer, TRUE);
        }
	else {
            *ppfOutput = fopen(sArg, "w");

	    if (*ppfOutput == NULL)
 	        FatalExit("Error opening specified file for output", 
                  FALSE);

	    bOutput = TRUE;
	}
    } /* for */

    if (!bEqn)
	FatalExit("Failed to specify an expression or polynomial", TRUE);
    if (!bInit)
        fprintf(stderr, "Assuming initial point is (0.0, 0.0, 0.0)\n");
    if (!bStep)
        fprintf(stderr, "Assuming step size is %.2f\n", DEF_GEOD_STEP);
    if (!bIterations)
        fprintf(stderr, "Assuming number of iterations per geodesic is %d\n",
          DEF_GEOD_ITERATIONS);
    if (!bGeodesics)
        fprintf(stderr, "Assuming number of geodesics is %d\n", DEF_GEODESICS);
    if (!bOutput)
        fprintf(stderr, "Assuming output goes to standard output\n");
    fprintf(stderr, "\n");

} /* ProcessArgs */



/*
 *   Function:	InteractiveInput
 *
 *   Purpose:	This function handles the interactive part of the user 
 *              interface for implicit geodesics.  It prompts the user for the 
 *		user-specified parameters and inputs these parameters.
 *
 *   Author:    Bret Johnson
 */

void InteractiveInput(pvctInitial, pscStep, pIterations, pGeodesics, peqn, 
  ppfOutput, psTitle, psEqn)
PVECTOR pvctInitial;
SCALAR	*pscStep;
short	*pIterations;
short	*pGeodesics;
PEQN	peqn;
FILE	**ppfOutput;
char	**psTitle;
char	**psEqn;
{
    char	sBuffer[MAX_STRING_LENGTH];	     /* String buffer */


    *psTitle = InputTitle(); 		/* Input the title */

    /* Input the initial point */
    Input("X coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[0]), "%lf");
    Input("Y coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[1]), "%lf");
    Input("Z coordinate of initial point [0.0]: ", 
        (char *) &(pvctInitial->vars[2]), "%lf");

    /* Input the step size */
    sprintf(sBuffer, "Step size [%.2f]: ", DEF_GEOD_STEP);
    Input(sBuffer, (char *) pscStep, "%lf");

    /* Input the number of iterations per geodesic */
    sprintf(sBuffer, "Number of iterations per geodesic [%d]: ", 
        DEF_GEOD_ITERATIONS);
    Input(sBuffer, (char *) pIterations, "%hd");

    sprintf(sBuffer, "Number of geodesics [%d]: ", DEF_GEODESICS);

    /* Loop until user enters valid number of geodesics */
    while (TRUE) {
        *pGeodesics = DEF_GEODESICS;
	Input(sBuffer, (char *) pGeodesics, "%hd");
	if (*pGeodesics > MAX_DIR || *pGeodesics < 1)
	    fprintf(stderr, 
              "The number of geodesics must be between 1 and %d.\n", MAX_DIR);
	else
	    break;
    }

    /* Input the equation type */
    peqn->et = InputEqnType();

    /* Input the expression or polynomial */
    if (peqn->et == E_EXPR)
        peqn->d.pexp = InputExpr(psEqn, '\0');
    else
        peqn->d.ppl = InputPoly(psEqn, '\0');

    /* Check that the initial point is on the surface */
    if (fabs(EvaluateEqn(peqn, pvctInitial)) > EPSILON)
        FatalExit("The initial point is not on the surface", FALSE);

    /* Input the file name where output is to go */
    *ppfOutput = InputOutputFile(
      "Name of file to hold output [standard output]: ", stdout);
}

