/*
*  Module:  parametric curvature lines user interface
*
*  Program: parcurv (parametric curvature lines generator for project Riemann)
*/

#define MAIN			/* Define global variables with this file */
#include "userinter.h"

static FILE * pfDevNull;	/* Pointer to file /dev/null */

/*
 *   Function:	UserInterface
 *
 *   Purpose:	This function handles the user interface for parametric 
 *		curvature lines.  It scans the command line for the appropriate
 *		parameters or prompts the user for them if the user didn't enter
 *		any command line parameters.  
 *
 *   Author:    Bret Johnson
 */

void UserInterface(argc, argv, pvctInitial, pscStep, pscSpacing, pscD1, pscD2,
  apeqn, ppf2dOutput, ppf3dOutput, psTitle, asEqn)
int     argc;
char    *argv[];
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR  *pscSpacing;
SCALAR	*pscD1, *pscD2;
PEQN    apeqn[3];
FILE    **ppf2dOutput, **ppf3dOutput;
char    **psTitle, *asEqn[3];
{
    int	i;


    if ( (pfDevNull = fopen("/dev/null", "w")) == NULL )
	FatalExit("Cannot open /dev/null for writing", FALSE);

    /* Set up default parameters */
    for (i = 0; i < NUMBER_OF_VARS; ++i) {
        pvctInitial->vars[i] = 0.0;
	apeqn[i]->et = DEF_EQNTYPE;
    }
    *pscStep = DEF_CURV_STEP;
    *pscSpacing = DEF_CURV_SPACING;
    *pscD1 = DEF_CURV_D1;
    *pscD2 = DEF_CURV_D2;
    *ppf2dOutput = pfDevNull;
    *ppf3dOutput = stdout;
    *psTitle = "";

    GlobalArgv = argv;
    GlobalArgc = argc;

    /* Form the usage error message */
    sprintf(sUsage, "Usage: %s [-t title] [-v initial_parameter_values] [-s \
step_size]\n\
  [-sp mesh_spacing] [-1 direction_one_distance] [-2 direction_two_distance]\n\
  [-2d 2D_output_file] [3D_output_file] <-ex x_expression -ey y_expression \n\
  -ez z_expression> | <-px x_polynomial -py y_polynomial -pz z_polynomial>", 
  GlobalArgv[0]);

    /* If the user entered no command line arguments, assume he wishes to */
    /*   use the interactive interface.  Otherwise, assume he is using */
    /*   the command line interface. */
    if (argc != 1)
        ProcessArgs(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, apeqn, 
          ppf2dOutput, ppf3dOutput, psTitle, asEqn);
    else InteractiveInput(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, apeqn,
           ppf2dOutput, ppf3dOutput, psTitle, asEqn);

} /* UserInterface */



/*
 *   Function:	ProcessArgs
 *
 *   Purpose:	This function processes all of the command line arguments.
 *
 *   Author:    Bret Johnson
 */

void ProcessArgs(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, apeqn, 
  ppf2dOutput, ppf3dOutput, psTitle, asEqn)
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR  *pscSpacing; 
SCALAR	*pscD1, *pscD2;
PEQN    apeqn[3];
FILE    **ppf2dOutput, **ppf3dOutput;
char    **psTitle, *asEqn[3];
{
    BOOL	bInit = FALSE,	     /* TRUE when initial point entered */
    		bStep = FALSE,	     /* TRUE when step size entered */
                bSpacing = FALSE,    /* TRUE when mesh spacing entered */
                bD1 = FALSE,         /* TRUE when direction one and two */
                bD2 = FALSE,         /*   distances entered */
          	b2dOutput = FALSE,   /* TRUE when 2d and 3d output files are */
          	b3dOutput = FALSE,   /*   entered */
    		bXPoly = FALSE,      /* TRUE when user inputs the x, y, and z */
	        bYPoly = FALSE,      /*   polynomials */
                bZPoly = FALSE,
    		bXExpr = FALSE,      /* TRUE when user inputs the x, y, and z */
	        bYExpr = FALSE,      /*   expressions */
                bZExpr = FALSE;
    char	*sArg;		     /* Current argument string */
    int		i;		     /* Argument counter */
    char	sBuffer[100];


    for (i = 1; i < GlobalArgc; ++i) {

        sArg = GlobalArgv[i];

        if (strcmp(sArg, "-t") == 0) {
	    ++i;
	    if (i >= GlobalArgc)
	        FatalExit("Failed to specify a title", TRUE);
	    *psTitle = GlobalArgv[i];
	}
	else if (strcmp(sArg, "-v") == 0) {
	    GetPointArg(2, pvctInitial, &i);	
	    bInit = TRUE;
	}
	else if (strcmp(sArg, "-s") == 0) {
	    GetStepSizeArg(pscStep, &i);
	    bStep = TRUE;
	}
	else if (strcmp(sArg, "-sp") == 0) {
            GetSpacingArg(pscSpacing, &i);
            bSpacing = TRUE;
	}
	else if (strcmp(sArg, "-1") == 0) {
            GetD1Arg(pscD1, &i);
            bD1 = TRUE;
	}
	else if (strcmp(sArg, "-2") == 0) {
            GetD2Arg(pscD2, &i);
            bD2 = TRUE;
	}
	else if (strcmp(sArg, "-ex") == 0) {
	    GetExprArg(apeqn[0], &(asEqn[0]), &i);
	    bXExpr = TRUE;
	}
	else if (strcmp(sArg, "-ey") == 0) {
	    GetExprArg(apeqn[1], &(asEqn[1]), &i);
	    bYExpr = TRUE;
	}
	else if (strcmp(sArg, "-ez") == 0) {
	    GetExprArg(apeqn[2], &(asEqn[2]), &i);
	    bZExpr = TRUE;
	}
	else if (strcmp(sArg, "-px") == 0) {
	    GetPolyArg(apeqn[0], &(asEqn[0]), &i);
	    bXPoly = TRUE;
	}
	else if (strcmp(sArg, "-py") == 0) {
	    GetPolyArg(apeqn[1], &(asEqn[1]), &i);
	    bYPoly = TRUE;
	}
	else if (strcmp(sArg, "-pz") == 0) {
	    GetPolyArg(apeqn[2], &(asEqn[2]), &i);
	    bZPoly = TRUE;
	}
	else if (strcmp(sArg, "-2d") == 0) {
	    ++i;
	    if (i >= GlobalArgc)
	        FatalExit("Failed to specify the 2D output file", TRUE);

            *ppf2dOutput = fopen(GlobalArgv[i], "w");

	    if (*ppf2dOutput == NULL)
 	        FatalExit("Error opening specified file for 2D output", 
                  FALSE);

	    b2dOutput = TRUE;
	}
	else  if (sArg[0] == '-' || b3dOutput) {
            sprintf(sBuffer, "Illegal option: \"%s\"", sArg);
	    FatalExit(sBuffer, TRUE);
        }
	else {
            *ppf3dOutput = fopen(sArg, "w");

	    if (*ppf3dOutput == NULL)
 	        FatalExit("Error opening specified file for 3D output", 
                  FALSE);

	    b3dOutput = TRUE;
	}
    } /* for */

    if ( ! ((bXPoly && bYPoly && bZPoly) || (bXExpr && bYExpr && bZExpr)) )
	FatalExit(
          "Failed to specify the x, y, and z expressions or polynomials", TRUE);
    if (!bInit)
        fprintf(stderr, "Assuming initial parameters are (0.0, 0.0)\n");
    if (!bStep)
        fprintf(stderr, "Assuming step size is %.2f\n", DEF_CURV_STEP);
    if (!bSpacing)
        fprintf(stderr, "Assuming mesh spacing is %.1f\n", DEF_CURV_SPACING);
    if (!bD1)
        fprintf(stderr, "Assuming direction one distance is %.1f\n",
          DEF_CURV_D1);
    if (!bD2)
        fprintf(stderr, "Assuming direction two distance is %.1f\n",
          DEF_CURV_D2);
    if (!b2dOutput)
	fprintf(stderr, "Assuming 2D output goes to /dev/null\n");
    if (!b3dOutput)
        fprintf(stderr, "Assuming 3D output goes to standard output\n");
    fprintf(stderr, "\n");

} /* ProcessArgs */



/*
 *   Function:	InteractiveInput
 *
 *   Purpose:	This function handles the interactive part of the user 
 *              interface for parametric curvature lines.  It prompts the user 
 *		for the user-specified parameters and inputs these parameters.
 *
 *   Author:    Bret Johnson
 */

void InteractiveInput(pvctInitial, pscStep, pscSpacing, pscD1, pscD2, apeqn,
  ppf2dOutput, ppf3dOutput, psTitle, asEqn)
PVECTOR pvctInitial;
SCALAR  *pscStep;
SCALAR  *pscSpacing;
SCALAR  *pscD1, *pscD2;
PEQN    apeqn[3];
FILE    **ppf2dOutput, **ppf3dOutput;
char    **psTitle, *asEqn[3];
{
    char	sBuffer[MAX_STRING_LENGTH];	     /* String buffer */
    int		i;


    *psTitle = InputTitle(); 		/* Input the title */

    /* Input the initial point */
    Input("Initial value of the s parameter [0.0]: ",
        (char *) &(pvctInitial->vars[0]), "%lf");
    Input("Initial value of the t parameter [0.0]: ", 
        (char *) &(pvctInitial->vars[1]), "%lf");

    /* Input the step size */
    sprintf(sBuffer, "Step size [%.2f]: ", DEF_CURV_STEP);
    Input(sBuffer, (char *) pscStep, "%lf");

    /* Input the mesh spacing */
    sprintf(sBuffer, "Mesh spacing [%.1f]: ", DEF_CURV_SPACING);
    Input(sBuffer, (char *) pscSpacing, "%lf");

    /* Input the direction one distance */
    sprintf(sBuffer, "Distance along direction one [%.1f]: ", DEF_CURV_D1);
    Input(sBuffer, (char *) pscD1, "%lf");

    /* Input the direction two distance */
    sprintf(sBuffer, "Distance along direction two [%.1f]: ", DEF_CURV_D2);
    Input(sBuffer, (char *) pscD2, "%lf");

    /* Input the equation type */
    apeqn[0]->et = InputEqnType();
    apeqn[1]->et = apeqn[0]->et;
    apeqn[2]->et = apeqn[0]->et;

    /* Input the expression or polynomial */
    if (apeqn[0]->et == E_EXPR)
	for (i = 0; i < 3; ++i)
            apeqn[i]->d.pexp = InputExpr(&(asEqn[i]), '1' + i);
    else
	for (i = 0; i < 3; ++i)
            apeqn[i]->d.ppl = InputPoly(&(asEqn[i]), '1' + i);

    /* Input the file names where 2D and 3D output is to go */
    *ppf2dOutput = InputOutputFile(
      "Name of file to hold 2D output [/dev/null]: ", pfDevNull);
    *ppf3dOutput = InputOutputFile(
      "Name of file to hold 3D output [standard output]: ", stdout);
}

