/*
 *  Module:   user interface
 *
 *  Programs: symgeod, numgeod, impcurv, and parcurv (project Riemann curvature
 *            lines and geodesics generators)
 */

#include "userinter.h"

/*
 *   Function:	Input
 *
 *   Purpose:	This function first prompts the user with message "sPrompt"
 *		and then has him enter a value of type indicated by "sSscanf".
 *		"sScanf" is a format specifier that will be passed to sscanf 
 *		to do the conversion.  This format specifier should only 
 *		cotain one type field as this function considers the input 
 *		valid if and only if sscanf says that it made one successful 
 *		conversion.  The prompting and entering is continued until the 
 *		user makes a valid input or simply enters a newline (thus 
 *		selecting the default).  If the user makes a valid input, it
 *		it is stored in *pInput.  If the user just presses enter, 
 *		*pInput's value is unchanged.
 *
 *   Author:    Bret Johnson
 */

void	Input(sPrompt, pInput, sSscanf)
char	sPrompt[];
char	*pInput;
char	sSscanf[];
{
    char    sBuffer[MAX_STRING_LENGTH];


    while (TRUE) {
        fprintf(stderr, "%s", sPrompt);     	/* Prompt the user */

	/* Input one line; if error in read or end of file, exit */
	if (!fgets(sBuffer, MAX_STRING_LENGTH, stdin))
	    FatalExit("Error in reading in string", FALSE);

 	if (sBuffer[0] == '\n')	 	/* If the user just typed enter, */
	    return;			/*   return */

	/* If the user entered a line beginning with a valid type, return */
	if (sscanf(sBuffer, sSscanf, pInput) == 1)
	    return;
    }
}



/*
 *   Function:	InputEqnType
 *
 *   Purpose:	This function continues to prompt the user for an equation type
 *		(either an expression or a polynomial) until he enters a valid
 *		one.  This function then returns the entered equation type.
 *
 *   Author:    Bret Johnson
 */

EQNTYPE InputEqnType()
{
    char	cEqnType;	/* Character user inputs for equation type */


    /* Loop until the user inputs a valid equation type */
    do {
        if (DEF_EQNTYPE == E_POLY) {
            cEqnType = 'p';	    
            Input(
             "Type of equation--either an (e)xpression or a (p)olynomial [p]: ",
              (char *) &cEqnType, "%c");
	}
        else {
	    cEqnType = 'e';
            Input(
             "Type of equation--either an (e)xpression or a (p)olynomial [e]: ",
              (char *) &cEqnType, "%c");
	}
    }
    while (cEqnType != 'e' && cEqnType != 'p');
	
    if (cEqnType == 'e')
        return(E_EXPR);
    else return(E_POLY);
}



/*
 *   Function:	InputPoly
 *
 *   Purpose:	This function continues to prompt the user for an polynomial
 *		until the user enters a valid (parsable) polynomial.  This
 *		function then parses the polynomial and returns a pointer to 
 *		the structure defininig the parsed polynomial.  This function 
 *		also returns, via its variable parameter, the string 
 *		representation of the polynomial.  polynum is a character that
 *		is the number of the polynomial displayed in the prompt.  For 
 *		example, if polynum is three, the prompt is "Polynomial 3: ".
 *		If polynum is the null terminator, the prompt is "Polynomial: ".
 *
 *   Author:    Bret Johnson
 */

PPOLY	InputPoly(psPoly, polynum)
char	**psPoly, polynum;
{
    PPOLY   ppl;	     			/* Parsed polynomial */
    char    sBuffer[MAX_STRING_LENGTH];
    BOOL    bSuccess;


    do {
        if (polynum != '\0')
            fprintf(stderr, "Polynomial %c: ", polynum);
	else
            fprintf(stderr, "Polynomial: ");

        bSuccess = fgets(sBuffer, MAX_STRING_LENGTH, stdin) && 
	  ( strlen(sBuffer) != MAX_STRING_LENGTH - 1 );

        /* Remove newline character */
        sBuffer[strlen(sBuffer) - 1] = '\0';

        if (bSuccess)
	    if (!(ppl = ParsePoly(sBuffer)))
		bSuccess = FALSE;
    }
    while (!bSuccess);

    *psPoly = MyMalloc((unsigned) (strlen(sBuffer) + 1));
    strcpy(*psPoly, sBuffer);
    return(ppl);
}



/*
 *   Function:	InputExpr
 *
 *   Purpose:	This function continues to prompt the user for an expression
 *		until the user enters a valid (parsable) expression.  This
 *		function then parses the expression, simplifies it,  and returns
 *              a pointer to the structure defininig the parsed expression.
 *		This function also returns, via its variable parameter, the
 *		string representation of the expression.  exprnum is a 
 *		character that is the number of the expression displayed in the
 *		prompt.  For example, if exprnum is three, the prompt is 
 *		"Expression 3: ".  If exprnum is the null terminator, the 
 *		prompt is "Expression: ".
 *
 *   Author:    Bret Johnson
 */

PEXPR InputExpr(psExpr, exprnum)
char	**psExpr, exprnum;
{
    PEXPR   pexp;	     /* Parsed expresssion */


    do
        if (exprnum != '\0')
            fprintf(stderr, "Expression %c: ", exprnum);
	else 
            fprintf(stderr, "Expression: ", exprnum);

    while (!(pexp = ReadExpr(stdin, psExpr)));


    return(pexp);
}
    

/*
 *   Function:	InputOutputFile
 *
 *   Purpose:	This function continues to prompt the user, with the passed 
 *		prompt, for the name of a file to hold output until the user 
 *              enters a valid file name (that is, a writable file) or just 
 *              presses enter to select the passed default file.  This function
 *              then opens the selected file, if the default is not chosen, for
 *              writing and returns a pointer to that file.
 *
 *   Author:    Bret Johnson
 */

FILE * InputOutputFile(sPrompt, pfDefault)
char * sPrompt;
FILE * pfDefault;
{
    FILE   *pfOutput;
    char    sBuffer[MAX_STRING_LENGTH];
    BOOL    bSuccess;


    do {
        fprintf(stderr, "%s", sPrompt);

        bSuccess = fgets(sBuffer, MAX_STRING_LENGTH, stdin) && 
	  ( strlen(sBuffer) != MAX_STRING_LENGTH - 1 );

	if (sBuffer[0] == '\n') {	/* Just pressing enter selects the */
	    pfOutput = pfDefault;	/*   default file */
	    break;
	}

        /* Remove newline character */
        sBuffer[strlen(sBuffer) - 1] = '\0';

        if (bSuccess)
	    if ( (pfOutput = fopen(sBuffer, "w")) == NULL )
		bSuccess = FALSE;
    }
    while (!bSuccess);

    return pfOutput;
}



/*
 *   Function:	InputTitle
 *
 *   Purpose:	This function prompts the user for the title to be stored in 
 *		the data file.  Titles longer than MAX_STRING_LENGTH produce an
 *		error.  This function returns the title string.
 *
 *   Author:    Bret Johnson
 */

char	*InputTitle()
{
    char	sBuffer[MAX_STRING_LENGTH];
    char       *sReturn;


    fprintf(stderr, "Title: ");
    if (!fgets(sBuffer, MAX_STRING_LENGTH, stdin) || strlen(sBuffer) ==
      MAX_STRING_LENGTH - 1)
        FatalExit("Error in reading in string", FALSE);

    /* Remove newline character */
    sBuffer[strlen(sBuffer) - 1] = '\0';

    sReturn = MyMalloc((unsigned) (strlen(sBuffer) + 1));
    strcpy(sReturn, sBuffer);

    return(sReturn);
}



/*
 *   Function:	FatalExit
 *
 *   Purpose:	This function prints the name of the program and the passed
 *		string to standard error and aborts the program, returning a one
 *		to indicate an error.  It should be used to exit under error 
 *		conditions.  If "bUsage" is TRUE then the usage string is 
 *		printed after the passed string is printed.
 *
 *   Author:    Bret Johnson
 */

void	FatalExit(s, bUsage)
char    *s;
BOOL     bUsage;
{
    fprintf(stderr, "%s: %s\n", GlobalArgv[0], s);
    if (bUsage) 
        fprintf(stderr, "%s\n", sUsage);
    fprintf(stderr, "\n");

    exit(1);
}



/*
 *   Function:	GetPointArg
 *
 *   Purpose:	This function reads in a point from the command line and returns
 *		it via pvctPoint.  dimensions is the number of coordinates 
 *		that the point should contain.  *pArg is the argument number
 *		of the "-v" argument, which signals the initial vector.  Upon 
 *              exit, *pArg is the argument number of the last coordinate of the
 *		point.  This function does error checking on the input point,
 *		doing a fatal exit if it is incorrectly specified.
 *
 *   Author:    Bret Johnson
 */

void	GetPointArg(dimensions, pvctPoint, pArg)
int	dimensions;
PVECTOR	pvctPoint;
int	*pArg;
{
    int		i = 0;		/* Counter of point dimensions */
    SCALAR	sc;
    char	sBuffer[100];


    ++(*pArg);		/* Skip past "-v" */

    while (TRUE) {

	/* If we have run out of arguments, break */
        if (*pArg >= GlobalArgc) {
            --(*pArg);
	    break;
        }

	if (sscanf(GlobalArgv[*pArg], "%lf", &sc) != 1) {
            --(*pArg);
	    break;
        }

        /* If the user entered too many coordinates, set i to an illegal */
        /*   amount and break */
        if (i >= dimensions) {
	    i += 10;
 	    break;
	}
        
	pvctPoint->vars[i++] = sc;
        ++(*pArg);
    }

    if (i != dimensions) {
	sprintf(sBuffer, "The initial point must have %d coordinates", 
          dimensions);
	FatalExit(sBuffer, TRUE);
    }
}



/*
 *   Function:	GetStepSizeArg
 *
 *   Purpose:	This function reads in the step size from the command line and
 *		returns it via pvctStepSize.  *pArg is the argument number
 *		of the "-s" argument, which signals the step size.  Upon 
 *              exit, *pArg is the argument number of the step size 
 *		specification.  This function does error checking on the step
 *		size and does a fatal exit if it is incorrectly specified.
 *
 *   Author:    Bret Johnson
 */

void	GetStepSizeArg(pscStepSize, pArg)
SCALAR	*pscStepSize;
int	*pArg;
{
    ++(*pArg);		/* Read past "-s" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy step size", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%lf", pscStepSize) != 1)
	FatalExit("Error in step size specification", TRUE);
}



/*
 *   Function:	GetIterationsArg
 *
 *   Purpose:	This function is the same as GetStepSizeArg, but instead gets 
 *		the number of iterations per geodesic from the command line.
 *
 *   Author:    Bret Johnson
 */

void	GetIterationsArg(pIterations, pArg)
short	*pIterations;
int	*pArg;
{
    ++(*pArg);		/* Read past "-i" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy number of iterations", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%hd", pIterations) != 1)
	FatalExit("Error in number of iterations per geodesic \
specification", TRUE);
}



/*
 *   Function:	GetGeodesicsArg
 *
 *   Purpose:	This function is the same as GetStepSizeArg, but instead gets
 *		the number of geodesics from the command line.  This function 
 *		ensures that the number of geodesics is between 1 and MAX_DIR.
 *
 *   Author:    Bret Johnson
 */

void	GetGeodesicsArg(pGeodesics, pArg)
short	*pGeodesics;
int	*pArg;
{
    char sBuffer[100];


    ++(*pArg);		/* Read past "-g" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy number of geodesics", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%hd", pGeodesics) != 1)
	FatalExit("Error in number of geodesics specification", TRUE);

    if (*pGeodesics > MAX_DIR || *pGeodesics < 1) {
	sprintf(sBuffer, "Number of geodesics must be between 1 and %d", 
            MAX_DIR);
        FatalExit(sBuffer, FALSE);
    }
}



/*
 *   Function:	GetSpacingArg
 *
 *   Purpose:	This function is the same as GetStepSizeArg, but instead gets 
 *		the mesh spacing from the command line.
 *
 *   Author:    Jim Lambers
 */

void GetSpacingArg(pscSpacing, pArg)
SCALAR	*pscSpacing;
int	*pArg;
{
    ++(*pArg);		/* Read past "-sp" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy mesh spacing", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%lf", pscSpacing) != 1)
	FatalExit("Error in mesh spacing specification", TRUE);
}



/*
 *   Function:	GetD1Arg
 *
 *   Purpose:	This function is the same as GetStepSizeArg, but instead gets 
 *		the distance for curvature lines in direction one from the 
 *		command line.
 *
 *   Author:    Bret Johnson
 */

void	GetD1Arg(pscD1, pArg)
SCALAR	*pscD1;
int	*pArg;
{
    ++(*pArg);		/* Read past "-1" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy direction one distance", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%lf", pscD1) != 1)
	FatalExit("Error in direction one distance specification", TRUE);
}



/*
 *   Function:	GetD2Arg
 *
 *   Purpose:	This function is the same as GetStepSizeArg, but instead gets 
 *		the distance for curvature lines in direction two from the 
 *		command line.
 *
 *   Author:    Bret Johnson
 */

void	GetD2Arg(pscD2, pArg)
SCALAR	*pscD2;
int	*pArg;
{
    ++(*pArg);		/* Read past "-2" */

    if (*pArg >= GlobalArgc)
	FatalExit("Failed to specifiy direction two distance", TRUE);

    if (sscanf(GlobalArgv[*pArg], "%lf", pscD2) != 1)
	FatalExit("Error in direction two distance specification", TRUE);
}



/*
 *   Function:	GetExprArg
 *
 *   Purpose:	This function is similar to GetStepSizeArg, but instead gets 
 *		an expression from the command line and returns it via peqn.
 *		The string representation of the expression (what the user 
 *		entered) is returned via psExpr.  The expression is simplified.
 *
 *   Author:    Jim Lambers
 */

void GetExprArg(peqn, psExpr, pArg)
PEQN	peqn;
char	**psExpr;
int	*pArg;
{
    peqn->et = E_EXPR;

    ++(*pArg);			/* Read past "-e" */

    if (*pArg >= GlobalArgc)
        FatalExit("Failed to specify an expression", TRUE);

    *psExpr = GlobalArgv[*pArg];
    peqn->d.pexp = ParseExpr(GlobalArgv[*pArg]);

    if (!(peqn->d.pexp))
        FatalExit("Expression not specified correctly", TRUE);

    SimplifyExpr(&(peqn->d.pexp));
}



/*
 *   Function:	GetPolyArg
 *
 *   Purpose:	This function is similar to GetStepSizeArg, but instead gets 
 *		a polynomial from the command line and returns it via peqn.
 *		The string representation of the polynomial (what the user 
 *		entered) is returned via psPoly.
 *
 *   Author:    Bret Johnson
 */

void GetPolyArg(peqn, psPoly, pArg)
PEQN	peqn;
char	**psPoly;
int	*pArg;
{
    peqn->et = E_POLY;
    ++(*pArg);			/* Read past "-p" */

    if (*pArg >= GlobalArgc)
        FatalExit("Failed to specify a polynomial", TRUE);

    *psPoly = GlobalArgv[*pArg];
    peqn->d.ppl = ParsePoly(GlobalArgv[*pArg]);

    if (!(peqn->d.ppl))
        FatalExit("Polynomial not specified correctly", TRUE);
}

