/*
*  Module:  compute transformation matrix
*
*  Program: xdisp (X11 grahpics displayer for project Riemann)
*/

#include "xdisp.h"

static	void	TransformPoint();

/*
*  Function: ComputeTransformMatrix
*
*  Purpose: This function computes the transformation matrix to project the
*           points in three space onto a plane and then map these points into
*	    an X window with the passed width and height.  The matrix is stored
*	    in the global variable T.
*
*  Author:  Bret Johnson
*/

BOOL ComputeTransformMatrix(psr, DeviceWindowWidth, DeviceWindowHeight)
PSURFACE  psr;
int       DeviceWindowWidth, DeviceWindowHeight;
{
    SCALAR 	W1[4][4], W2[4][4], W3[4][4];	/* Work matrices */
    SCALAR	sc1, sc2, sc3, sc4;	       	/* Work variables */
    SCALAR	scSx, scSy;     /* Scaling values for mapping into device */
				/*   coordinates */
    SCALAR	scTx, scTy;     /* Translation values for mapping into */
				/*   device coordinates */
    SCALAR 	scThetaInRadians;	/* Angles are stored in degrees & */
    SCALAR 	scEtaInRadians;		/*   translated into radians when */
					/*   used */
    SCALAR	scXProjMin, scZProjMin;	/* Minimum and maximum world */
    SCALAR	scXProjMax, scZProjMax;	/*   coordinates in projection plane */
    POINT	pt;
    int		i;
    SCALAR	scWorldWindowWidth, 	/* Size of world coordinate window */
    		scWorldWindowHeight;	
    int		DeviceSubwindowWidth,   /* Size (in pixels) of part of device */
		DeviceSubwindowHeight;	/*   coordinate window to which */
					/*   surface is mapped */

 
    /* Make W1 a matrix which translates VRP to the origin */
    FILL4x4(W1,   1.0, 		0.0, 		0.0,		0.0,
		  0.0, 		1.0, 		0.0, 		0.0,
		  0.0, 		0.0, 		1.0, 		0.0,
		  -VRP.x, 	-VRP.y, 	-VRP.z, 	1.0);

    scThetaInRadians = scTheta * (2.0 * PI / 360.0);
    sc1 = cos(-scThetaInRadians - PI/2.0);
    sc2 = sin(-scThetaInRadians - PI/2.0);
    sc3 = -sin(-scThetaInRadians - PI/2.0);           
    sc4 = cos(-scThetaInRadians - PI/2.0);

    /* Note: The comments describing the following transformation matrices */
    /* assume a perspective projection.  For a parallel projection, one can */
    /* think of there being a COP (the center of projection or the observer's */
    /* eye) located an infinite distance from the view plane (thus scD = */
    /* infinity).  */

    /* Make W2 a matrix which rotates points about the z axis; this matrix */
    /*   rotates the COP to the yz coordinate plane.  The COP will be rotated */
    /*   to the negative y half of the plane assuming scD is positive and -90 */
    /*   degrees < scEta < 90 degrees. */
    FILL4x4(W2,	sc1,	sc2,	0.0,	0.0,
		sc3,	sc4,	0.0,	0.0,
		0.0,	0.0,	1.0,	0.0,
		0.0,	0.0,	0.0,	1.0);

    MultiplyMatrices(W1, W2, W3);

    scEtaInRadians = scEta * (2.0 * PI / 360.0);
    sc1 = cos(scEtaInRadians);
    sc2 = sin(scEtaInRadians);
    sc3 = -sin(scEtaInRadians);
    sc4 = cos(scEtaInRadians);

    /* Make W1 a matrix which rotates points about the x axis; this matrix */
    /*   rotates the COP (already in the yz coordinate plane) to be on the */
    /*   negative y axis (assuming scD is positive) */
    FILL4x4(W1,	1.0,	0.0,	0.0,	0.0,
		0.0,	sc1,	sc2,	0.0,
		0.0,	sc3,	sc4,	0.0,
		0.0,	0.0,	0.0,	1.0);

    MultiplyMatrices(W3, W1, W2);

    /* Make W1 a matrix which does a perspective or parallel projection on */
    /*   the projection plane; the projection plane is the xz coordinate */
    /*   plane */
    if (!bParallel) {
        FILL4x4(W1, 1.0,	0.0,	0.0,	0.0,
		    0.0,	0.0,	0.0,	1.0/scD,
		    0.0,	0.0,	1.0,	0.0,
		    0.0,	0.0,	0.0,	1.0);
    }
    else {
        FILL4x4(W1, 1.0,	0.0,	0.0,	0.0,
		    0.0,	0.0,	0.0,	0.0,
		    0.0,	0.0,	1.0,	0.0,
		    0.0,	0.0,	0.0,	1.0);
    }

    MultiplyMatrices(W2, W1, W3);

    /* Now W3 transforms world 3D coordinates into world 2D coordinates on */
    /*   the projection plane.  We must now find the minimum and maximum */
    /*   2D world coordinates in the projection plane.  To do this we */
    /*   transform the eight corner points of the right rectangular prism */
    /*   enclosing all of the 3D points and use the minimum and maximum of */
    /*   the obtained X and Y coordinates to define a rectangle enclosing all */
    /*   of the projected points */

    TransformPoint(psr->XMin, psr->YMin, psr->ZMin, W3, &pt);
    scXProjMin = pt.x;
    scXProjMax = pt.x;
    scZProjMin = pt.z;
    scZProjMax = pt.z;

    /* Test other seven points */
    for (i = 2; i <= 8; ++i) {

        switch(i) {
 	    case 2:TransformPoint(psr->XMin, psr->YMin, psr->ZMax, W3, &pt);
		   break;
 	    case 3:TransformPoint(psr->XMin, psr->YMax, psr->ZMin, W3, &pt);
		   break;
 	    case 4:TransformPoint(psr->XMin, psr->YMax, psr->ZMax, W3, &pt);
		   break;
 	    case 5:TransformPoint(psr->XMax, psr->YMin, psr->ZMin, W3, &pt);
		   break;
 	    case 6:TransformPoint(psr->XMax, psr->YMin, psr->ZMax, W3, &pt);
		   break;
 	    case 7:TransformPoint(psr->XMax, psr->YMax, psr->ZMin, W3, &pt);
		   break;
 	    case 8:TransformPoint(psr->XMax, psr->YMax, psr->ZMax, W3, &pt);
		   break;
	}

        scXProjMin = MIN(pt.x, scXProjMin);
        scXProjMax = MAX(pt.x, scXProjMax);
        scZProjMin = MIN(pt.z, scZProjMin);
        scZProjMax = MAX(pt.z, scZProjMax);

    } /* for */

    /* Compute the world coordinate window size */
    scWorldWindowWidth  = scXProjMax - scXProjMin;
    scWorldWindowHeight = scZProjMax - scZProjMin;

    /* If either world coordinate window dimension is zero, change that */
    /*   dimension to a small number */

    if (scWorldWindowWidth == 0) {
	scWorldWindowWidth = EPSILON;
	scXProjMin -= EPSILON / 2.0;
	scXProjMax += EPSILON / 2.0;
    }

    if (scWorldWindowHeight == 0) {
	scWorldWindowHeight = EPSILON;
	scZProjMin -= EPSILON / 2.0;
	scZProjMax += EPSILON / 2.0;
    }

    /* Shrink the device coordinate width & height to allow for a border */
    DeviceSubwindowWidth = DeviceWindowWidth - 2 * PictureBorderWidth;
    DeviceSubwindowHeight = DeviceWindowHeight - 2 * PictureBorderWidth;

    /* Ensure that the surface will fit in the window */
    if (DeviceSubwindowWidth <= 0 || DeviceSubwindowHeight <= 0)
	return FALSE;

    /* If the user wishes to preserve the aspect ratio, then shrink the */
    /*   device subwindow width or height so that the device subwindow has */
    /*   the same aspect ratio as the world coordinate window */
    if (bPAR)
        if ( (SCALAR) DeviceSubwindowWidth / scWorldWindowWidth > 
          (SCALAR) DeviceSubwindowHeight / scWorldWindowHeight )

	    DeviceSubwindowWidth = ROUND( scWorldWindowWidth * 
	      (SCALAR) DeviceSubwindowHeight / scWorldWindowHeight );

        else DeviceSubwindowHeight = ROUND( scWorldWindowHeight * 
               (SCALAR) DeviceSubwindowWidth / scWorldWindowWidth );

    /* Compute scale and translation parameters */
    scSx = (SCALAR) DeviceSubwindowWidth / scWorldWindowWidth;
    scSy = - (SCALAR) DeviceSubwindowHeight / scWorldWindowHeight;
    scTx = -scXProjMin * scSx + 
        (SCALAR) (DeviceWindowWidth - DeviceSubwindowWidth) / 2.0;
    scTy = scZProjMax * (- scSy) + 
        (SCALAR) (DeviceWindowHeight - DeviceSubwindowHeight) / 2.0;

    /* Make W1 a matrix which maps the 2D world coordinates into device */
    /*   coordinates */
    FILL4x4(W1,	scSx,	0.0,	0.0,	0.0,
		0.0,	0.0,	0.0,	0.0,
		0.0,	scSy,	0.0,	0.0,
		scTx,	scTy,	0.0,	1.0);

    MultiplyMatrices(W3, W1, T);

    if (bVerbose)
	PrintMatrix(T, "Transformation matrix:");

    return TRUE;
}



/*
*  Function: TransformPoint
*
*  Purpose: This function transforms the passed point using the passed matrix.
*           This function simply multiplies a row vector times a matrix.
*
*  Author:  Bret Johnson
*/

static void TransformPoint(x, y, z, Tr, pptResult)
SCALAR	x, y, z;
SCALAR	Tr[4][4];		/* Transformation matrix */
PPOINT	pptResult; 		/* Result vector (the transformed point) */
{
    SCALAR	scWeight;

    pptResult->x = x*Tr[0][0] + y*Tr[1][0] + z*Tr[2][0] + Tr[3][0];
    pptResult->y = x*Tr[0][1] + y*Tr[1][1] + z*Tr[2][1] + Tr[3][1];
    pptResult->z = x*Tr[0][2] + y*Tr[1][2] + z*Tr[2][2] + Tr[3][2];
    scWeight = x*Tr[0][3] + y*Tr[1][3] + z*Tr[2][3] + Tr[3][3];

    pptResult->x /= scWeight;		/* Adjust so point has weight one */
    pptResult->y /= scWeight;
    pptResult->z /= scWeight;
}

