/*
*  Module:  draw surface
*
*  Program: xdisp (X11 grahpics displayer for project Riemann)
*/

#include "xdisp.h"

static void ParallelTransformPoints();
static void PerspectiveTransformPoints();

/*
*  Function: DrawSurface
*
*  Purpose: This function clears the passed drawable, computes the 
*	    transformation matix, and draws the passed surface on it.
*
*  Author:  Bret Johnson
*/

void DrawSurface(dr, psr)
Drawable 	dr;
PSURFACE	psr;
{
    XPoint	axpt[MAX_POINTS];	/* Temporary array of transformed */
					/*    points */
    PLINE	pln;


    /* Clear the drawable */
    XSetForeground(pds, gcPicture, pxlBackground);
    XFillRectangle(pds, dr, gcPicture, 0, 0, PictureWindowWidth, 
        PictureWindowHeight);

    /* Compute the transformation matrix; if we should not draw anything, */
    /*   return */
    if ( ComputeTransformMatrix(psr, PictureWindowWidth, PictureWindowHeight) ==
      FALSE )
        return;

    /* Set the drawing color for geodesics */
    XSetForeground(pds, gcPicture, pxlColorG);

    for (pln = psr->plnFirst; pln != NULL; pln = pln->plnNext) {

	/* If drawing curvature lines, set line color */
	if (IS_CURVATURE_LINES(psr->type))
	    if (pln->direction == 1)
                XSetForeground(pds, gcPicture, pxlColor1);
	    else XSetForeground(pds, gcPicture, pxlColor2);

        if (bParallel)
            ParallelTransformPoints(pln->papt, pln->length, axpt);
	else 
            PerspectiveTransformPoints(pln->papt, pln->length, axpt);

        XDrawLines(pds, dr, gcPicture, axpt, pln->length, CoordModeOrigin);

 	/* Connect 1st & last points of curv. lines if asked to */
	if (IS_CURVATURE_LINES(psr->type))
	    if (pln->direction == 1 && bConnect1)
		XDrawLine(pds, dr, gcPicture, (int) axpt[0].x, (int) axpt[0].y,
		  (int) axpt[pln->length - 1].x, (int) axpt[pln->length - 1].y);
	    else if (pln->direction == 2 && bConnect2)
		XDrawLine(pds, dr, gcPicture, (int) axpt[0].x, (int) axpt[0].y, 
		  (int) axpt[pln->length - 1].x, (int) axpt[pln->length - 1].y);
    }
}



/*
*  Function: ParallelTransformPoints
*
*  Purpose: This function transforms the npoints 3D points in array apt and 
*           stores the transformed 2D points in array axpt.  The points are
*	    transformed with the global matrix T, assumed to do a parallel
*	    projection.
*
*  Author:  Bret Johnson
*/

static void ParallelTransformPoints(apt, npoints, axpt)
POINT	apt[];
int	npoints;
XPoint	axpt[];
{
    register    int 	i;


    for (i = 0; i < npoints; ++i) {
	axpt[i].x = ROUND( (apt[i].x)*T[0][0] + (apt[i].y)*T[1][0] + T[3][0] );
	axpt[i].y = ROUND( (apt[i].x)*T[0][1] + (apt[i].y)*T[1][1] + 
          (apt[i].z)*T[2][1] + T[3][1] );
    }
}



/*
*  Function: PerspectiveTransformPoints
*
*  Purpose: This function transforms the npoints 3D points in array apt and 
*           stores the transformed 2D points in array axpt.  The points are
*	    transformed with the global matrix T, assumed to do a perspective
*	    projection.
*
*  Author:  Bret Johnson
*/

static void PerspectiveTransformPoints(apt, npoints, axpt)
POINT	apt[];
int	npoints;
XPoint	axpt[];
{
    register    int 	i;
    register	SCALAR	x, y, z;	/* Input point */
    SCALAR	xt, yt, weight;         /* Transformed point */


    for (i = 0; i < npoints; ++i) {

	x = apt[i].x;
	y = apt[i].y;
	z = apt[i].z;

	xt = x*T[0][0] + y*T[1][0] + z*T[2][0] + T[3][0];
	yt = x*T[0][1] + y*T[1][1] + z*T[2][1] + T[3][1];
	weight = x*T[0][3] + y*T[1][3] + z*T[2][3] + T[3][3];

        /* Adjust so weight is one */
        axpt[i].x = ROUND(xt / weight);
        axpt[i].y = ROUND(yt / weight);
    }
}

