/*
*  Module:  process options
*
*  Program: xdisp (X11 grahpics displayer for project Riemann)
*/

#include "control.h"
#include <X11/Xresource.h>

static void      SetOptionVariables();

static char     *sDisplayName = NULL;
static char     *sColor1, *sColor2, *sColorG, *sBackground;
static char	*sFont;
static BOOL      bFileName = FALSE,   /* These boolean variables are TRUE if */
    		 bColor1 = FALSE,     /*   the user sets the associated value */
    		 bColor2 = FALSE,     /*   on the command line */
    		 bColorG = FALSE,
    		 bBackground = FALSE,
    		 bTheta = FALSE,
		 bEta = FALSE,
		 bDistance = FALSE,
		 bFont = FALSE;


/*
*  Function: ProcessOptions
*
*  Purpose: This function process the command line and sets most parameters to
*           their correct values.  In general, if a parameter is not set via 
*	    the command line then the resource database is checked.  If it is 
*	    not specified in the resource database then the default value is 
*	    used.  argc and argv are the same as passed to main.  ppfData is 
*	    used to return a file pointer to the data file.  psGeometry is used
*	    to return the user's command line picture window geometry string.
*	    If no such geometry string is specified on the command line, then 
*	    *psGeometry is set to NULL.  psControlGeometry is the same as 
*	    psGeometry except that it is for the control window geometry.
*
*  Author:  Bret Johnson
*/

void ProcessOptions(argc, argv, ppfData, psGeometry, psControlGeometry) 
int	  argc; 
char	 *argv[];
FILE	**ppfData;
char	**psGeometry, **psControlGeometry;
{
    int	         i;
    char	*sFileName;


    /* Neither geometry string has yet been entered */
    *psGeometry = NULL;
    *psControlGeometry = NULL;

    /* Set some parameters to their defaults */
    bDebug = FALSE;
    bVerbose = FALSE;
    bNoDraw = FALSE;

    /* Process command line arguments */
    for (i = 1; i < argc; i++) {

        if (strcmp(argv[i], "-display") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify display name"); 
            sDisplayName = argv[i];
	}

        else if (strcmp(argv[i], "-db") == 0)
	    bDebug = TRUE;

        else if (strcmp(argv[i], "-vr") == 0)
	    bVerbose = TRUE;

        else if (strcmp(argv[i], "-nd") == 0)
	    bNoDraw = TRUE;

        else if (strcmp(argv[i], "-cl1") == 0) {
	    bColor1 = TRUE;
            if (++i >= argc)
 	        FatalExit(TRUE, 
                  "Failed to specify curvature line direction one color");
            sColor1 = argv[i];
        }

        else if (strcmp(argv[i], "-cl2") == 0) {
	    bColor2 = TRUE;
            if (++i >= argc)
 	        FatalExit(TRUE, 
                  "Failed to specify curvature line direction two color");
            sColor2 = argv[i];
        }

        else if (strcmp(argv[i], "-clg") == 0) {
	    bColorG = TRUE;
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify color of geodesics");
            sColorG = argv[i];
        }

        else if (strcmp(argv[i], "-bg") == 0) {
	    bBackground = TRUE;
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify background color"); 
            sBackground = argv[i];
        }

        else if (strcmp(argv[i], "-t") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify angle theta"); 
            if ( sscanf(argv[i], "%lf", &scTheta) != 1 )
 	        FatalExit(TRUE, "Incorrect specification of angle theta"); 
	    bTheta = TRUE;
	}

        else if (strcmp(argv[i], "-e") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify angle eta"); 
            if ( sscanf(argv[i], "%lf", &scEta) != 1 )
 	        FatalExit(TRUE, "Incorrect specification of angle eta"); 
            bEta = TRUE;
	}

        else if (strcmp(argv[i], "-d") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, 
                  "Failed to specify distance from projection plane"); 
            if ( sscanf(argv[i], "%lf", &scD) != 1 )
 	        FatalExit(TRUE, "Incorrect specification of distance \
from projection plane");
            bDistance = TRUE;
	}

        else if (strcmp(argv[i], "-fn") == 0) {
	    bFont = TRUE;
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify font");
            sFont = argv[i];
        }

	else if (strcmp(argv[i], "-geometry") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify geometry"); 
	    *psGeometry = argv[i];
	}

	else if (strcmp(argv[i], "-cgeometry") == 0) {
            if (++i >= argc)
 	        FatalExit(TRUE, "Failed to specify control window geometry"); 
	    *psControlGeometry = argv[i];
	}

	else if (!bFileName && *argv[i] != '-') {
            sFileName = argv[i];
	    bFileName = TRUE;
	}

	else
  	    FatalExit(TRUE, "Illegal option \"%s\"", argv[i]);

    } /* for */

    /* Open the specified file; if the user didn't provide a file name, use */
    /*   standard input */
    if (bFileName)
	if ((*ppfData = fopen(sFileName, "r")) == NULL)
            FatalExit(FALSE, "Cannot access file \"%s\"", sFileName);
	else ;
    else
	*ppfData = stdin;

    SetOptionVariables();
}



/*
*  Function: SetOptionVariables
*
*  Purpose: This function sets the values for most of the parameter variables.
*	    In general, if a value was not specified on the command line then
*	    the resource database is checked.  If a value is not specified 
*	    there then the default value is used.
*
*  Author:  Bret Johnson
*/

static void SetOptionVariables()
{
    char	*s;		/* Work string */


    if ((pds = XOpenDisplay(sDisplayName)) == NULL)
        FatalExit(FALSE, "Cannot open display \"%s\"", sDisplayName);

    screen = DefaultScreen(pds);
    cm = DefaultColormap(pds, screen);

    /* If the user did not set the colors from the command line, check the */
    /*   resource database; if the colors are not specified in either of */
    /*   these two locations, use black or white as defaults */

    if (!bColor1)
        if ( (sColor1 = XGetDefault(pds, sProgramName, "Color1")) == NULL )
            sColor1 = "black";
    pxlColor1 = AllocNamedColor(sColor1);

    if (!bColor2)
        if ( (sColor2 = XGetDefault(pds, sProgramName, "Color2")) == NULL )
            sColor2 = "black";
    pxlColor2 = AllocNamedColor(sColor2);

    if (!bColorG)
        if ( (sColorG = XGetDefault(pds, sProgramName, "ColorGeodesics")) == 
          NULL )
            sColorG = "black";
    pxlColorG = AllocNamedColor(sColorG);

    if (!bBackground)
        if ( (sBackground = XGetDefault(pds, sProgramName, "Background")) == 
          NULL )
            sBackground = "white";
    pxlBackground = AllocNamedColor(sBackground);

    /* Check if the connect parameters are set in the resource database; set */
    /*   the associated variables to TRUE if and only if these parameters are */
    /*   set to on */

    bConnect1 = FALSE;
    if ( (s = XGetDefault(pds, sProgramName, "Connect1")) != NULL )
        bConnect1 = (strcmp(s, "on") == 0);

    bConnect2 = FALSE;
    if ( (s = XGetDefault(pds, sProgramName, "Connect2")) != NULL )
        bConnect2 = (strcmp(s, "on") == 0);

    if ( (s = XGetDefault(pds, sProgramName, "Connect")) != NULL )
        if (strcmp(s, "on") == 0) {
	    bConnect1 = TRUE;
            bConnect2 = TRUE;
	}

    /* If the user did not set the theta, eta, or distance parameters from */
    /*   the command line, check the resource database; if these parameters */
    /*   are not specified in either of these two locations, use program */
    /*   defaults */

    if (!bTheta)
        if ( (s = XGetDefault(pds, sProgramName, "Theta")) != NULL )
	    if (sscanf(s, "%lf", &scTheta) != 1)
		FatalExit(FALSE,
                  "Bad specification of theta in resource database file");
	    else ;
	else scTheta = -90.0;

    /* Adjust scTheta so that is is between -180 and 180 */
    scTheta = (SCALAR) ( ROUND(scTheta) % 360 );
    if (scTheta > 180.0)
	scTheta -= 360.0;
    else if (scTheta < -180.0)
	scTheta += 360.0;

    if (!bEta)
        if ( (s = XGetDefault(pds, sProgramName, "Eta")) != NULL )
	    if (sscanf(s, "%lf", &scEta) != 1)
		FatalExit(FALSE,
                  "Bad specification of eta in resource database file");
	    else ;
	else scEta = 0.0;

    /* Adjust scEta so that is is between -180 and 180 */
    scEta = (SCALAR) ( ROUND(scEta) % 360 );
    if (scEta > 180.0)
	scEta -= 360.0;
    else if (scEta < -180.0)
	scEta += 360.0;

    bUSDistance = TRUE;
    if (!bDistance)
        if ( (s = XGetDefault(pds, sProgramName, "Distance")) != NULL )
	    if (sscanf(s, "%lf", &scD) != 1)
		FatalExit(FALSE, "Bad specification of distance in resource \
database file");
	    else ;
	else bUSDistance = FALSE;

    /* Check if the preserve aspect ratio, parallel projection, and auto */
    /*   redraw parameters are set in the resource database; if these */
    /*   parameters are set, use the parameter value; otherwise, use the */
    /*   default value */

    bPAR = TRUE;
    if ( (s = XGetDefault(pds, sProgramName, "PreserveAspectRatio")) != NULL )
        bPAR = (strcmp(s, "on") == 0);

    bParallel = FALSE;
    if ( (s = XGetDefault(pds, sProgramName, "ParallelProjection")) != NULL )
        bParallel = (strcmp(s, "on") == 0);

    bAutoRedraw = FALSE;
    if ( (s = XGetDefault(pds, sProgramName, "AutoRedraw")) != NULL )
        bAutoRedraw = (strcmp(s, "on") == 0);

    bEqnTitle = FALSE;
    if ( (s = XGetDefault(pds, sProgramName, "EquationTitle")) != NULL )
        bEqnTitle = (strcmp(s, "on") == 0);

    /* If the font was not specified on the command line, check the resoure */
    /*   database; if the font is specified in neither of these places, use */
    /*   the 8x13 font */
    if (!bFont)
        if ( (sFont = XGetDefault(pds, sProgramName, "Font")) == NULL )
            sFont = "8x13";
    pxfs = XLoadQueryFont(pds, sFont);
    if (pxfs == 0)
        FatalExit(FALSE, "Cannot load font \"%s\"", sFont);
}

