/*
*  Module:  read surface
*
*  Program: xdisp (X11 grahpics displayer for project Riemann)
*/

#include "xdisp.h"

static void   ReadPointsInLine();
static void   EatLine();
static char  *MyFgets();
static void   ReadString();


/*
*  Function: ReadSurface
*
*  Purpose: This function reads in the surface in data file pfData and stores 
*	    the surface is psr.
*
*  Author:  Bret Johnson
*/

void ReadSurface(pfData, psr)
FILE     *pfData;
PSURFACE  psr;
{
    BOOL    bBegin = TRUE;	/* TRUE when reading in first point */
    PLINE   pln;
    POINT   aptBuffer[MAX_POINTS];
    char    sBuffer[MAX_STRING_LENGTH];
    int     LookAhead;


    /* Read in the type */
    ReadString(pfData, "Type: ", sBuffer);
    if (strcmp(sBuffer, "implicit geodesics") == 0)
	psr->type = IMP_GEOD;
    else if (strcmp(sBuffer, "implicit curvature lines") == 0)
	psr->type = IMP_CURV;
    else if (strcmp(sBuffer, "parametric geodesics") == 0)
	psr->type = PAR_GEOD;
    else if (strcmp(sBuffer, "parametric curvature lines") == 0)
	psr->type = PAR_CURV;
    else FatalExit(FALSE, "Bad type in input data");

    /* If this is parametric data, read in the type of space */
    psr->bParSpace = FALSE;
    if (IS_PARAMETRIC(psr->type)) {
        ReadString(pfData, "Space: ", sBuffer);
        if (sBuffer[0] == 'p')
	    psr->bParSpace = TRUE;
	else if (sBuffer[0] != 'n')
	    FatalExit(FALSE, "Illegal type of space in input data");
    }

    /* Read in the title */
    ReadString(pfData, "Title: ", sBuffer);
    psr->sTitle = malloc( (unsigned int) (strlen(sBuffer) + 1) );
    strcpy(psr->sTitle, sBuffer);

    /* Read in the creation date & time */
    ReadString(pfData, "Created: ", sBuffer);
    psr->sCreated = malloc( (unsigned int) (strlen(sBuffer) + 1) );
    strcpy(psr->sCreated, sBuffer);

    /* Read in the equation type */
    ReadString(pfData, "Equation type: ", sBuffer);
    if (sBuffer[0] == 'p')
	psr->et = E_POLY;
    else if (sBuffer[0] == 'e')
	psr->et = E_EXPR;
    else FatalExit(FALSE, "Illegal equation type in input data");

    /* Read in the implicit equation or parametric equations */
    if (IS_IMPLICIT(psr->type)) {
	ReadString(pfData, "Implicit equation: ", sBuffer);
        psr->sEquation1 = malloc( (unsigned int) (strlen(sBuffer) + 1) );
        strcpy(psr->sEquation1, sBuffer);
    }
    else {
	ReadString(pfData, "Parametric equation 1: ", sBuffer);
        psr->sEquation1 = malloc( (unsigned int) (strlen(sBuffer) + 1) );
        strcpy(psr->sEquation1, sBuffer);

	ReadString(pfData, "Parametric equation 2: ", sBuffer);
        psr->sEquation2 = malloc( (unsigned int) (strlen(sBuffer) + 1) );
        strcpy(psr->sEquation2, sBuffer);
	
	ReadString(pfData, "Parametric equation 3: ", sBuffer);
        psr->sEquation3 = malloc( (unsigned int) (strlen(sBuffer) + 1) );
        strcpy(psr->sEquation3, sBuffer);
    }

    /* Eat up miscellaneous information; an empty line signals the end of */
    /*   such information */
    while (TRUE) {
	LookAhead = getc(pfData);
	if (LookAhead == '\n' || LookAhead == EOF)
	    break;
	EatLine(pfData);
    }

    psr->LineCount = 0;
    while (TRUE) {

	if ( (LookAhead = getc(pfData)) == EOF )
	    break;

	ungetc(LookAhead, pfData);

        /* Allocate space for new line */
        if (bBegin) {
            psr->plnFirst = (PLINE) malloc(sizeof(LINE));
            pln = psr->plnFirst;
	}
	else {
	    pln->plnNext = (PLINE) malloc(sizeof(LINE));
	    pln = pln->plnNext;
	}

        if (IS_CURVATURE_LINES(psr->type)) {
	    ReadString(pfData, "Line in direction ", sBuffer);
    
	    if (sBuffer[0] == '1')
	        pln->direction = 1;
	    else if (sBuffer[0] == '2')
	        pln->direction = 2;
	    else FatalExit(FALSE, "Bad line direction in input data");
	}
	else (ReadString(pfData, "Line:", sBuffer));

	++(psr->LineCount);

	ReadPointsInLine(pfData, psr, pln, aptBuffer, &bBegin);

    } /* while */

    if (bBegin)
	FatalExit(FALSE, "Input contains no line data");

    pln->plnNext = NULL;

    /* The view reference point is located in the center of the surface */
    VRP.x = (psr->XMin + psr->XMax) / 2.0;
    VRP.y = (psr->YMin + psr->YMax) / 2.0;
    VRP.z = (psr->ZMin + psr->ZMax) / 2.0;

    /* Set the surface size to the distance between opposite corners of the */
    /*   right rectangular prism which encloses the surface */
    psr->scSurfaceSize = sqrt( (psr->XMax - psr->XMin)*(psr->XMax - psr->XMin) +
      (psr->YMax - psr->YMin)*(psr->YMax - psr->YMin) +
      (psr->ZMax - psr->ZMin)*(psr->ZMax - psr->ZMin) );

    /* Don't allow the surface size to be zero */
    if (psr->scSurfaceSize == 0.0)
	FatalExit(FALSE, "Surface is just a single point");

    fclose(pfData);

} /* ReadSurface */



/*
*  Function: ReadPointsInLine
*
*  Purpose: This function reads points from pf and stores them in pln as a 
*	    LINE.  aptBuffer must point to an array of MAX_POINTS points to be 
*	    used as a buffer.  psr is the surface associated with the line.
*	    pbBegin must point to TRUE or FALSE depending on whether or not a 
*	    point has yet been read in.  This function sets *pbBegin to TRUE 
*	    as soon as it reads in a point.  Note that points in the 
*	    parameter space as treated as points in three space with zero as
*	    their z coordinate.   Thus a parallel projection of these points
*	    from straight above the x-y plane is equivalent to graphing the
*	    points in two dimensions.
*
*  Author:  Bret Johnson
*/

static void ReadPointsInLine(pf, psr, pln, aptBuffer, pbBegin)
FILE	  *pf;
PSURFACE   psr;
PLINE	   pln;
POINT	   aptBuffer[];
BOOL	  *pbBegin;		/* TRUE if first point yet to be read in */
{
    int		i;
    int	    	result;


    for (i = 0; TRUE; ++i) {

        if (i >= MAX_POINTS)
	    FatalExit(FALSE, "Lines cannot consist of more than %d points", 
              MAX_POINTS);

	/* Read in two coordinates for parameter space or three coordinates */
	/*   for normal space; note that one is added to "result" in the 2D */
	/*   case so that in both cases result != 3 signals the end of list */
        if (psr->bParSpace) {
            result = fscanf(pf, "%lf%lf", &aptBuffer[i].x, &aptBuffer[i].y);
	    ++result;
	    aptBuffer[i].z = 0.0;
	}
        else result = fscanf(pf, "%lf%lf%lf", &aptBuffer[i].x, &aptBuffer[i].y, 
	  &aptBuffer[i].z);

        if (result != 3) {

	    /* Lines with no points are not allowed */
	    if (i == 0)
	        FatalExit(FALSE, "Curve number %d in input data \
contains no points", psr->LineCount);

	    /* When a line has one point, that point is copied to make all */
	    /*   lines have at least two points */
	    else if (i == 1) {
		i = 2;
		aptBuffer[1].x = aptBuffer[0].x;
		aptBuffer[1].y = aptBuffer[0].y;
		aptBuffer[1].z = aptBuffer[0].z;
	    }

            pln->length = i;

	    /* Store points in data structure */
            pln->papt = (PPOINT) malloc( (unsigned int) (sizeof(POINT) * i) );
            bcopy( (char *) aptBuffer, (char *) (pln->papt), 
              (unsigned int) sizeof(POINT) * i);

	    return;
        }

	EatLine(pf);        /* Eat up any trailing spaces and the newline */

	if (*pbBegin) {
	    psr->XMin = psr->XMax = aptBuffer[i].x;
	    psr->YMin = psr->YMax = aptBuffer[i].y;
	    psr->ZMin = psr->ZMax = aptBuffer[i].z;
            *pbBegin = FALSE;
	}	
        else {
            psr->XMin = MIN(psr->XMin, aptBuffer[i].x);
            psr->YMin = MIN(psr->YMin, aptBuffer[i].y);
            psr->ZMin = MIN(psr->ZMin, aptBuffer[i].z);
            psr->XMax = MAX(psr->XMax, aptBuffer[i].x);
            psr->YMax = MAX(psr->YMax, aptBuffer[i].y);
            psr->ZMax = MAX(psr->ZMax, aptBuffer[i].z);
        }

    } /* while */
}



/*
*  Function: EatLine
*
*  Purpose: This function reads characters from file pf and throws them away
*	    up to and including a newline.
*
*  Author:  Bret Johnson
*/

static void EatLine(pf)
FILE	*pf;
{
    while (getc(pf) != '\n')
       ;
}



/*
*  Function: MyFgets
*
*  Purpose: This function is the same as the standard fgets except that 
*           an error message is printed if the input line is too long (as 
*	    compared to the passed buffer size) and the trailing spaces on the
* 	    input line (including the newline) are removed.
*/

static char * MyFgets(s, n, pf)
char	*s;
int	n;
FILE	*pf;
{
    char 	*result;
    int		i;


    result = fgets(s, n, pf);

    /* If error or end of file, return NULL and empty string */
    if (result == NULL) {
        s[0] = '\0';
        return NULL;
    }

    if (strlen(s) == n-1)
        FatalExit(FALSE, 
          "Data file lines must not be longer than %d characters.", n-3);

    /* Remove trailing spaces and newline */
    for (i = strlen(s) - 1; i >= 0; --i)
	if (isspace(s[i]))
	    s[i] = '\0';
	else break;

    return result;
}



/*
*  Function: ReadString
*
*  Purpose: This function reads in the next line from the file pf.  It compares
*	    the first part of the read-in string to ensure that it matches
*	    sPrompt and error exits if there is no match.  The part of the 
*	    read-in string after the matched prefix is then returned via
*	    sResult.
*
*  Author:  Bret Johnson
*/

static void ReadString(pf, sPrompt, sResult)
FILE	*pf;
char	*sPrompt;
char	*sResult;
{
    char	*sBuffer;
    int		 i;


    sBuffer = malloc( (unsigned int) (strlen(sPrompt) + MAX_STRING_LENGTH + 
      10) );

    MyFgets(sBuffer, MAX_STRING_LENGTH, pf);

    /* Find last non space character in sPrompt */
    for (i = strlen(sPrompt) - 1; sPrompt[i] == ' '; --i)
        ;

    /* If prompt minus trailing spaces does not match first part of input, */
    /*   error exit */
    if ( strncmp(sPrompt, sBuffer, i+1) != 0 )
	FatalExit(FALSE, "Didn't find \"%s\" in input data", sPrompt);

    strcpy( sResult, &(sBuffer[strlen(sPrompt)]) );

    free(sBuffer);
}

